/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings.ksp

import com.google.devtools.ksp.getDeclaredFunctions
import com.google.devtools.ksp.getDeclaredProperties
import com.google.devtools.ksp.isConstructor
import com.google.devtools.ksp.isPublic
import com.google.devtools.ksp.processing.CodeGenerator
import com.google.devtools.ksp.processing.Dependencies
import com.google.devtools.ksp.processing.KSPLogger
import com.google.devtools.ksp.processing.Resolver
import com.google.devtools.ksp.processing.SymbolProcessor
import com.google.devtools.ksp.symbol.ClassKind
import com.google.devtools.ksp.symbol.KSAnnotated
import com.google.devtools.ksp.symbol.KSClassDeclaration
import com.google.devtools.ksp.symbol.KSDeclaration
import com.google.devtools.ksp.symbol.KSFunctionDeclaration
import com.google.devtools.ksp.symbol.KSName
import com.google.devtools.ksp.symbol.KSPropertyDeclaration
import com.google.devtools.ksp.symbol.KSTypeReference
import com.google.devtools.ksp.symbol.KSValueParameter
import com.google.devtools.ksp.symbol.Modifier
import com.squareup.kotlinpoet.FileSpec
import com.squareup.kotlinpoet.ksp.writeTo
import ru.auroraos.kmp.qtbindings.ksp.export.*
import ru.auroraos.kmp.qtbindings.ksp.gen.*

private const val qtExport = "ru.auroraos.kmp.qtbindings.QtExport"

internal class QtBindingsSymbolProcessor(
    private val codeGenerator: CodeGenerator,
    private val logger: KSPLogger,
    private val options: QtBindingsOptions,
) : SymbolProcessor {
    private var isHelperKotlinCodeGenerated = false
    private val spec = ExportSpec()
    private val kotlinSpec = ExportKotlinSpec(codeGenerator)
    private lateinit var resolver: Resolver

    fun <T : KSDeclaration> Sequence<T>.qtExportFiler() = filter {
        if (it.parentDeclaration != null) {
            logger.warn("Annotation $qtExport can only be applied to the top-level classes and functions", it)
            return@filter false
        }

        if (!it.isPublic()) {
            logger.warn("Annotation $qtExport can only be applied to public classes and functions", it)
            return@filter false
        }

        // The main reason for not supporting expected declarations is that if an expected
        // and an actual declaration are annotated with a QtExport annotation, the generated
        // Qt files will contain two identical declarations, which is not allowed.
        if (it.isExpect) {
            logger.warn("Annotation $qtExport can not be applied to the expected classes and functions", it)
            return@filter false
        }

        if (it.typeParameters.any()) {
            logger.warn("Annotation $qtExport can not be applied to the generic classes and functions", it)
            return@filter false
        }

        if (it is KSClassDeclaration && it.classKind == ClassKind.OBJECT) {
            logger.warn("Annotation $qtExport is not supported for objects", it)
            return@filter false
        }

        return@filter true
    }

    override fun process(resolver: Resolver): List<KSAnnotated> {
        this.resolver = resolver // Unsure whether the resolver can be changed between KSP rounds, so we're updating it
        val deferredSymbols = mutableListOf<KSAnnotated>()

        // Find all exported symbols (should be public, top-level, not generic, not actual functions and classes)
        val exportedSymbols = resolver.getSymbolsWithAnnotation(qtExport)
        val classes = exportedSymbols.filterIsInstance<KSClassDeclaration>().qtExportFiler()
        val functions = exportedSymbols.filterIsInstance<KSFunctionDeclaration>().qtExportFiler()

        // Save Kotlin classes that user wants to export
        val (deferredClasses, savedClasses) = declareUserClasses(classes.asSequence())
        deferredSymbols.addAll(deferredClasses)

        // Process kotlin top-level classes and function
        functions.forEach { processExportedTopLevelFunction(it) ?: deferredSymbols.add(it) }
        savedClasses.forEach { processExportedClass(it) ?: deferredSymbols.add(it) }

        if (deferredSymbols.isEmpty()) {
            kotlinSpec.generate()
            generateHelperKotlinCode()
        }

        return deferredSymbols
    }

    private fun processExportedTopLevelFunction(f: KSFunctionDeclaration): ExportedFunction? {
        val file = f.containingFile?.let {
            spec.getFileOrCreate(it.fqName) {
                ExportedFile(
                    it.fileName.removeKotlinExtension(),
                    it.filePath,
                    it.packageName.asPackage(),
                )
            }
        } ?: run {
            logger.warn("Top-level function is not contained in the source file", f)
            return null
        }

        val exportFunc = processExportedFunction(f, file)
        if (exportFunc != null) {
            file.functions.add(exportFunc)
        }

        return exportFunc
    }

    private fun processExportedFunction(
        f: KSFunctionDeclaration, file: ExportedFile
    ): ExportedFunction? {
        val extensionReceiver = f.extensionReceiver
        var receiverParameter: ExportedParameter? = null
        val name = f.simpleName.getShortName()

        val returnType = f.returnType?.toExportedType() ?: run {
            logger.warn("Failed to resolve function return type", f)
            return null
        }

        if (extensionReceiver != null) {
            val receiverType = extensionReceiver.toExportedType() ?: run {
                logger.warn("Failed to resolve extension receiver type", f)
                return null
            }

            receiverParameter = ExportedParameter("thiz", receiverType)
        }

        val parameters = f.parameters.mapNotNull { param ->
            val exportedParam = processExportedParameter(param)
            if (exportedParam == null) {
                logger.warn("Failed to resolve function parameter", f)
            }

            exportedParam
        }

        if (parameters.size != f.parameters.size) {
            // Failed parameters has been already logged
            return null
        }

        val isAsync = f.isSuspend

        if (isAsync) {
            val funSpec = f.toCSuspendFunction()
            val containingFile = f.containingFile

            if (funSpec == null) {
                logger.warn("Failed to wrap suspend function", f)
                return null
            }

            if (containingFile == null) {
                logger.warn("Failed to get containing file for function", f)
                return null
            }

            kotlinSpec.getFileSpecBuilder(containingFile).addFunction(funSpec)
        }

        val isExtension = receiverParameter != null
        val finalParameters = if (!isExtension) parameters else listOf(receiverParameter) + parameters
        return ExportedFunction(name, isAsync, isExtension, finalParameters, returnType, file)
    }

    private fun processExportedClass(c: KSClassDeclaration): ExportedClass? {
        val exportedClass = c.toExportedClass() ?: return null

        if (exportedClass.isDataClass) {
            val containingFile = c.containingFile ?: run {
                logger.warn("Failed to get containing file os the class", c)
                return null
            }
            kotlinSpec.getFileSpecBuilder(containingFile).addFunction(generateCopyNoArgs(c))
        }

        val methods = c.getDeclaredPublicMethods()
        val properties = c.getDeclaredPublicProperties()
        val exportedMethods = methods.mapNotNull { processExportedMethod(it, exportedClass) }
        val exportedProperties = properties.flatMap { processExportedProperty(it, exportedClass) }
        exportedClass.methods.addAll(exportedMethods)
        exportedClass.properties.addAll(exportedProperties)
        return exportedClass
    }

    private fun processExportedProperty(
        property: KSPropertyDeclaration, parent: ExportedClass
    ): List<ExportedProperty> {
        // Invariants: public, not expected (because expected classes are not supported), not top-level
        // TODO: top-level properties is not supported yet

        if (property.typeParameters.any()) {
            logger.warn("Properties with type parameters is not supported", property)
            return listOf()
        }

        val props = mutableListOf<ExportedProperty>()
        val name = property.simpleName.asString()
        val extensionReceiver = property.extensionReceiver
        var extensionReceiverParameter: ExportedParameter? = null
        val type = property.type.toExportedType()

        if (type == null) {
            logger.warn("Failed to resolve property type", property)
            return listOf()
        }

        if (extensionReceiver != null) {
            val receiverType = extensionReceiver.toExportedType() ?: run {
                logger.warn("Failed to resolve extension receiver type", property)
                return listOf()
            }

            extensionReceiverParameter = ExportedParameter("thiz", receiverType)
        }

        if (property.getter != null) {
            props.add(ExportedProperty(name, type, AccessorType.Getter, parent, extensionReceiverParameter))
        }

        if (property.isMutable && property.setter != null) {
            props.add(ExportedProperty(name, type, AccessorType.Setter, parent, extensionReceiverParameter))
        } else if (property.isMutable && property.setter == null) {
            logger.warn("Failed to get property setter", property)
        }

        if (props.isEmpty()) {
            logger.warn("Failed to get property getter and setter", property)
            return listOf()
        }

        return props
    }

    private fun processExportedMethod(m: KSFunctionDeclaration, parent: ExportedClass): ExportedMethod? {
        // Invariants: public, not expected (because expected classes are not supported), not top-level

        if (m.typeParameters.any()) {
            logger.warn("Methods with type parameters is not supported", m)
            return null
        }

        val isConstructor = m.isConstructor()
        val exportedFunc = processExportedFunction(m, parent.containingFile) ?: return null
        return ExportedMethod(
            exportedFunc.originalKotlinName,
            isConstructor,
            exportedFunc.isAsync,
            exportedFunc.isExtension,
            parent,
            exportedFunc.returnType,
            exportedFunc.parameters,
        )
    }

    private fun processExportedParameter(parameter: KSValueParameter): ExportedParameter? {
        // Vararg and default parameter are not supported
        // The default parameter is exported, but becomes non-default
        if (parameter.isVararg) return null
        val type = parameter.type.toExportedType() ?: return null
        val name = parameter.name?.asString() ?: return null
        return ExportedParameter(name, type)
    }

    override fun finish() {
        // Resolve types, cNames and write hpp, cpp files
        val cApi = CLibraryAPI(options.libName)
        val typeResolver = QtResolver(cApi, resolver)
        val qtFiles = typeResolver.resolve(spec)

        val generators = listOf(Pair(QtHeaderGenerator(cApi), "hpp"), Pair(QtSourceGenerator(cApi), "cpp"))
        val kspFiles = resolver.getAllFiles()

        qtFiles.forEach { file ->
            val packageName = file.namespace.components.joinToString(".")
            val kspFile = kspFiles.firstOrNull { it.filePath == file.dependencyFilePath }

            generators.forEach { (generator, extension) ->
                val dependency = if (kspFile != null) Dependencies(true, kspFile) else Dependencies(false)
                codeGenerator.createNewFile(dependency, packageName, file.fileName, extension).bufferedWriter()
                    .use { writer ->
                        val t = generator.generate(file)
                        writer.write(t)
                    }
            }
        }

        generateQtRuntime()
    }

    private fun declareUserClasses(userClasses: Sequence<KSClassDeclaration>): Pair<List<KSClassDeclaration>, List<KSClassDeclaration>> {
        val deferred = mutableListOf<KSClassDeclaration>()
        val resolved = mutableListOf<KSClassDeclaration>()
        userClasses.forEach { if (it.toExportedClass() == null) deferred.add(it) else resolved.add(it) }
        return Pair(deferred, resolved)
    }

    private fun KSClassDeclaration.toExportedClass(): ExportedClass? {
        // Invariant: public, top-level, not-expected, not generic

        val ksFile = containingFile

        if (ksFile == null) {
            logger.warn("Failed to find file for class", this)
            return null
        }

        // Create or get exported containing file
        val containingFile = spec.getFileOrCreate(ksFile.fqName) {
            val fileName = ksFile.fileName.removeKotlinExtension()
            val filePackage = packageName.asPackage()
            ExportedFile(fileName, ksFile.filePath, filePackage)
        }

        // Create or get exported class and add it to the file
        val name = simpleName.getShortName()
        val fqName = qualifiedName!!.asString() // It's top-level class, so it must exist
        val isDataClass = Modifier.DATA in modifiers
        val exportedClass = spec.getClassOrCreate(fqName, { ExportedClass(name, isDataClass, containingFile) })

        if (!containingFile.classes.contains(exportedClass)) containingFile.classes.add(exportedClass)
        return exportedClass
    }

    private fun KSTypeReference.toExportedType() = toExportedType(spec.userClasses)

    private fun generateHelperKotlinCode() {
        if (!isHelperKotlinCodeGenerated) {
            val builder = FileSpec.builder("ru.auroraos.kmp.qtbindings.generated", "Generated")
            builder.addFunctions(generateListFunctionSpecs())
            builder.build().writeTo(codeGenerator, false)
            isHelperKotlinCodeGenerated = true
        }
    }

    private fun generateQtRuntime() {
        val qtGenerator = QtRuntimeGenerator()
        qtGenerator.generate().forEach { generatedFile ->
            if (generatedFile.fileContent == null) {
                logger.error("Failed to generate Qt runtime file ${generatedFile.fileName}.${generatedFile.fileExtension}")
            } else {
                val file = codeGenerator.createNewFile(
                    Dependencies(false),
                    generatedFile.packageName,
                    generatedFile.fileName,
                    generatedFile.fileExtension,
                )

                file.bufferedWriter().use { it.write(generatedFile.fileContent) }
            }
        }
    }
}

private fun KSClassDeclaration.getDeclaredPublicMethods(): Sequence<KSFunctionDeclaration> {
    // We do not export inherited methods for now
    return this.getDeclaredFunctions().filter { it.isPublic() }
}

private fun KSClassDeclaration.getDeclaredPublicProperties(): Sequence<KSPropertyDeclaration> {
    // We do not export inherited properties for now
    return this.getDeclaredProperties().filter { it.isPublic() }
}

private fun KSName.asPackage(): ExportedPackage {
    return ExportedPackage(this.asString().split('.').map { it.trim() })
}
