/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings.ksp

import com.google.devtools.ksp.getFunctionDeclarationsByName
import com.google.devtools.ksp.isConstructor
import com.google.devtools.ksp.isPublic
import com.google.devtools.ksp.processing.Resolver
import com.google.devtools.ksp.symbol.FunctionKind
import com.google.devtools.ksp.symbol.KSClassDeclaration
import com.google.devtools.ksp.symbol.KSFile
import com.google.devtools.ksp.symbol.KSFunctionDeclaration
import ru.auroraos.kmp.qtbindings.ksp.export.*

/**
 * Helps to convert Kotlin function and property names to C library ones.
 */
internal class KotlinScope(
    val spec: ExportSpec, val components: List<String>, val resolver: Resolver
) {
    private val names: MutableSet<String> = mutableSetOf()
    private val functionToUniqueCName: MutableMap<ExportedFunction, String> = mutableMapOf()

    fun cName(f: ExportedFunction): String {
        functionToUniqueCName[f]?.let { return it }

        val isTopLevel = f !is ExportedMethod
        val functionKind = if (isTopLevel) FunctionKind.TOP_LEVEL else FunctionKind.MEMBER

        /*
        *   Overloading functions in the same scope are exported to C by adding n * '_' to the name. For example, if
        *   there are 3 public functions func(par1, par2), func(par1), func(par1, par2, par3), they name will be mapped
        *   in the following way:
        *       * func(par1, par2) -> func(par1, par2)
        *       * func(par1) -> func_(par1)
        *       * func(par1, par2, par3) -> func__(par1, par2, par3)
        *
        *   For this reason, the name of a function in the C interface depends on the order in which the functions were
        *   processed by the compiler. And perhaps the order of processing is the most obscure point. I noticed that the
        *   order of function processing by the compiler and Sequence returned by KSP are the same. Let's hope that it
        *   will remain so from now on.
        *
        *   Exporting functions names to C Interface: https://github.com/JetBrains/kotlin/blob/96f142cf7d357f0f17a75a785234944b3f4dd4d5/kotlin-native/backend.native/compiler/ir/backend.native/src/org/jetbrains/kotlin/backend/konan/cexport/CAdapterGenerator.kt#L117
        * */

        resolver.getFunctionDeclarationsByName(f.fqName, isTopLevel)
            .filter { it.functionKind == functionKind && it.isPublic() }.forEach {
                // Get unique name for symbol and add it the uniqueNames set
                val uniqueName = getUniqueName(it.cSimpleName)

                // Try to find ExportedFunction that is mapped to KSFunctionDeclaration
                val containingFile = it.containingFile ?: return@forEach

                // containingFile is generated by qtbindings-ksp or created by user
                it.getExportedFunction(containingFile, isTopLevel)
                    ?.let { f -> functionToUniqueCName.put(f, uniqueName) }
            }

        return functionToUniqueCName.getOrElse(f) { getUniqueName(f.cSimpleName) }
    }


    fun cName(p: ExportedProperty): String {
        return when (p.accessor) {
            AccessorType.Getter -> "get_${p.name}"
            AccessorType.Setter -> "set_${p.name}"
        }
    }

    private fun KSFunctionDeclaration.toTopLevelFunction(file: ExportedFile) =
        file.functions.firstOrNull { isSameAs(it) }

    private fun KSFunctionDeclaration.getExportedFunction(
        containingFile: KSFile, isTopLevel: Boolean
    ): ExportedFunction? {

        // Try to get by file path
        val userExportedFile = spec.getFile(containingFile.fqName)
        if (userExportedFile != null) {
            if (isTopLevel) toTopLevelFunction(userExportedFile) else toClassMethod(userExportedFile)
        }

        // Could not find by file path, so the function is generated by us, it should be async function
        val parent = parentDeclaration
        val packageName = containingFile.packageName.asString()
        if (parent != null) {
            // It's an async method, try to find by class name
            val fqClassName = "$packageName.${parent.simpleName.asString()}"
            val possibleFile = spec.userClasses.firstOrNull { it.fqName == fqClassName }?.containingFile
            return possibleFile?.let { toClassMethod(it) }
        } else {
            // It's a top-level async function, search for it in all exported files
            return spec.userFiles.flatMap { it.functions }.firstOrNull {
                if (it.containingFile.filePackage.name != packageName) return@firstOrNull false
                return@firstOrNull isSameAs(it)
            }
        }
    }

    private fun KSFunctionDeclaration.toClassMethod(file: ExportedFile): ExportedMethod? {
        val parentClass = this.parentDeclaration as? KSClassDeclaration ?: return null
        val className = parentClass.simpleName.asString()
        val exportedClass = spec.userClasses.find { it.name == className } ?: return null
        return exportedClass.methods.firstOrNull { exportedMethod ->
            if (isConstructor() != exportedMethod.isConstructor) return@firstOrNull false
            return@firstOrNull isSameAs(exportedMethod)
        }
    }

    private fun KSFunctionDeclaration.isSameAs(exportedFunc: ExportedFunction): Boolean {
        if (simpleName.asString() != exportedFunc.name) return false
        if ((extensionReceiver != null) != exportedFunc.isExtension) return false
        // We can't check for suspend, because we wrap suspend functions and make them non-suspend
        // if (isSuspend != exportedFunc.isAsync) return false

        val exportedFuncParams =
            if (exportedFunc.isExtension) exportedFunc.parameters.drop(1) else exportedFunc.parameters

        if (parameters.size != exportedFuncParams.size) return false

        // Match parameter types
        if (parameters.mapNotNull { it.toExportParameter(spec.userClasses) } != exportedFuncParams) return false

        // There is no need to check return type as overloading function
        // can be identified by name and parameters
        return true
    }

    // Get unique name and add it immediately
    private fun getUniqueName(name: String): String {
        var uniqueName = name
        while (names.contains(uniqueName)) {
            uniqueName += "_"
        }
        names.add(uniqueName)
        return uniqueName
    }
}

private val KSFunctionDeclaration.cSimpleName: String
    get() {
        // TODO: check for CName annotation
        val parentClass = parentDeclaration as? KSClassDeclaration
        return if (isConstructor() && parentClass != null) parentClass.simpleName.asString() else simpleName.asString()
    }

// Fallback for case when some error has occurred
private val ExportedFunction.cSimpleName: String
    get() = if (this is ExportedMethod && isConstructor) parentClass.name else name
