/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings.gradle.plugin

import com.google.devtools.ksp.gradle.KspExtension
import org.gradle.api.GradleException
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.jetbrains.kotlin.gradle.dsl.KotlinMultiplatformExtension
import org.jetbrains.kotlin.gradle.plugin.KotlinSourceSet
import org.jetbrains.kotlin.gradle.plugin.KotlinTarget
import org.jetbrains.kotlin.gradle.plugin.mpp.KotlinNativeTarget
import org.jetbrains.kotlin.konan.target.Family

public class QtBindingsPlugin : Plugin<Project> {

    private companion object {
        val kmpPluginId = "org.jetbrains.kotlin.multiplatform"
        val kspPluginId = "com.google.devtools.ksp"
        val kspQtBindingsDependencyId = "ru.auroraos.kmp:qtbindings-ksp"
        val annotationsQtBindingsDependency = "ru.auroraos.kmp:qtbindings-annotations"
        val coreQtBindingsDependency = "ru.auroraos.kmp:qtbindings-core"
    }

    override fun apply(project: Project) {
        val qtBindings = project.extensions.create("qtBindings", QtBindingsExtension::class.java)

        project.pluginManager.withPlugin(kmpPluginId) {
            val kmp = project.extensions.getByType(KotlinMultiplatformExtension::class.java)
            val commonMainSourceSet = kmp.sourceSets.getByName(KotlinSourceSet.COMMON_MAIN_SOURCE_SET_NAME)
            project.configurations.getByName(commonMainSourceSet.implementationConfigurationName).dependencies.apply {
                add(project.dependencies.create("$annotationsQtBindingsDependency:$VERSION"))
                add(project.dependencies.create("$coreQtBindingsDependency:$VERSION"))
            }

            project.pluginManager.withPlugin(kspPluginId) {
                val ksp = project.extensions.getByType(KspExtension::class.java)

                kmp.targets.configureEach { target ->
                    if (!target.isLinuxTarget) return@configureEach

                    project.dependencies.add(
                        "ksp${target.capitalizedTargetName}", "$kspQtBindingsDependencyId:$VERSION"
                    )
                    ksp.arg(QtBindingsCommandLineArgumentProvider(qtBindings))
                }
            }
        }

        project.checkKmpAndKsp()
    }

    private fun Project.checkKmpAndKsp() {
        afterEvaluate {
            if (!project.plugins.hasPlugin(kmpPluginId)) {
                throw GradleException("QtBindings gradle plugin has been applied without KMP plugin")
            }

            if (!project.plugins.hasPlugin(kspPluginId)) {
                throw GradleException("QtBindings gradle plugin has been applied without KSP plugin")
            }
        }
    }
}

private val KotlinTarget.isLinuxTarget: Boolean
    get() = this is KotlinNativeTarget && konanTarget.family == Family.LINUX

private val KotlinTarget.capitalizedTargetName: String
    get() = targetName.replaceFirstChar { it.uppercaseChar() }
