/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings

import kotlinx.cinterop.*
import kotlinx.coroutines.CancellationException
import ru.auroraos.kmp.qtbindings.cruntime.*
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch

private val defaultCoroutineScope = CoroutineScope(Dispatchers.Default + SupervisorJob())

@OptIn(ExperimentalForeignApi::class)
private fun Job.asCCancelFunction(): CCancelCoroutine = staticCFunction { thiz ->
    thiz ?: return@staticCFunction
    thiz.pointed.coroutine?.asStableRef<Job>()?.get()?.cancel()
}

@OptIn(ExperimentalForeignApi::class)
private fun returnCString(callback: COnStringCallback, context: COpaquePointer, str: String) {
    memScoped {
        // cString must be copied before using it from native side
        val cString = str.cstr.getPointer(this)
        callback(context, cString)
    }
}

@OptIn(ExperimentalForeignApi::class)
internal fun returnCLong(callback: COnResultCallback, context: COpaquePointer, long: Long) {
    callback(context, long.toCPointer())
}

@OptIn(ExperimentalForeignApi::class)
public fun <T> cSuspend(
    coroutineScope: CoroutineScope = defaultCoroutineScope, block: suspend () -> T
): CPointer<KotlinCoroutineLauncher> {
    val stableRefPair = StableRef.create(Pair(coroutineScope, block))
    val cCoroutineLauncherFunc = staticCFunction {
            cContext: COpaquePointer,
            kotlinContext: COpaquePointer,
            onResult: COnResultCallback,
            onError: COnErrorCallback,
            onCancelled: COnCancelledCallback,
        ->

        val pair = kotlinContext.asStableRef<Pair<CoroutineScope, suspend () -> T>>()
        val (scope, suspendedBlock) = pair.get()

        val job = scope.launch {
            try {
                val result = suspendedBlock()
                when (result) {
                    null, is Unit -> {
                        onResult(cContext, null)
                    }

                    is String -> {
                        @Suppress("UNCHECKED_CAST")
                        returnCString(onResult as COnStringCallback, cContext, result)
                    }

                    is Boolean -> returnCLong(onResult, cContext, result.toByte().toLong())
                    is Char -> returnCLong(onResult, cContext, result.code.toLong())
                    is Byte -> returnCLong(onResult, cContext, result.toLong())
                    is Short -> returnCLong(onResult, cContext, result.toLong())
                    is Int -> returnCLong(onResult, cContext, result.toLong())
                    is Long -> returnCLong(onResult, cContext, result)
                    is UByte -> returnCLong(onResult, cContext, result.toLong())
                    is UShort -> returnCLong(onResult, cContext, result.toLong())
                    is UInt -> returnCLong(onResult, cContext, result.toLong())
                    is ULong -> returnCLong(onResult, cContext, result.toLong())
                    is Float -> returnCLong(onResult, cContext, result.toBits().toLong())
                    is Double -> returnCLong(onResult, cContext, result.toBits().toLong())

                    else -> {
                        // Result has to be disposed from native side when it's no longer needed
                        val stableResult = StableRef.create(result)
                        onResult(cContext, stableResult.asCPointer())
                    }
                }
            } catch (e: CancellationException) {
                throw e
            } catch (e: Throwable) {
                returnCString(onError, cContext, e.toString())
            }
        }

        // Handle cancel exception outside the coroutine scope
        job.invokeOnCompletion { cause ->
            if (cause !is CancellationException) return@invokeOnCompletion

            val exceptionMessage = cause.message
            if (exceptionMessage == null) onCancelled(cContext, null) else returnCString(
                onCancelled, cContext, exceptionMessage
            )
        }

        val stableJob = StableRef.create(job)
        val cancellableCoroutine = nativeHeap.alloc<CCancellableCoroutine>()
        cancellableCoroutine.coroutine = stableJob.asCPointer()
        cancellableCoroutine.cancel = job.asCCancelFunction()
        cancellableCoroutine.free = staticCFunction { thiz ->
            thiz ?: return@staticCFunction
            // Dispose Job and free CCancellableCoroutine allocated in nativeHeap
            thiz.pointed.coroutine?.asStableRef<Job>()?.dispose()
            nativeHeap.free(thiz)
        }
        return@staticCFunction cancellableCoroutine
    }

    val launcher = nativeHeap.alloc<KotlinCoroutineLauncher>()
    launcher.kotlinContext = stableRefPair.asCPointer()
    launcher.coroutineLauncherFunc = cCoroutineLauncherFunc.reinterpret()
    launcher.free = staticCFunction { thiz ->
        thiz ?: return@staticCFunction
        // Dispose kotlinContext: Pair<CoroutineScope, suspend () -> T>
        // Free KotlinCoroutineLauncher allocated in nativeHeap
        thiz.pointed.kotlinContext?.asStableRef<Pair<CoroutineScope, suspend () -> T>>()?.dispose()
        nativeHeap.free(thiz)
    }

    return launcher.ptr
}
