/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings

import kotlinx.cinterop.*

// Can be used for MutableList as List is a parent for MutableList
@OptIn(ExperimentalForeignApi::class)
public fun listGetSizePredefined(ptr: COpaquePointer?): Int {
    ptr ?: return 0
    val list = ptr.asStableRef<List<*>>()
    return list.get().size
}

// Can be used for MutableList as List is a parent for MutableList
@OptIn(ExperimentalForeignApi::class)
public fun listGetElementByIndexPredefined(ptr: COpaquePointer?, index: Int): COpaquePointer? {
    ptr ?: return null
    val list = ptr.asStableRef<List<*>>().get() // Ptr is stable ref already
    if (index < 0 || index >= list.size) return null
    val element = list[index] ?: return null
    return element.toOpaquePointer()
}

// MutableList can be used where list is needed as List is a parent for MutableList
public fun mutableListCreateEmptyPredefined() : MutableList<Any> {
    return mutableListOf()
}

// Can be used only for classes
@OptIn(ExperimentalForeignApi::class)
public fun mutableListAddClassPredefined(l: MutableList<Any>, ptr: COpaquePointer) {
    l.add(ptr.asStableRef<Any>().get())
}

// Should be used for simple types
public fun <T : Any> mutableListAddSimpleElementPredefined(l: MutableList<T>, el : T) {
    l.add(el)
}

public fun mutableListToListPredefined(ml: MutableList<*>) : List<*> {
    return ml.toList()
}

@OptIn(ExperimentalForeignApi::class)
private fun Any.toOpaquePointer() : COpaquePointer? {
    return when (this) {
        is Unit -> null
        is Boolean -> this.toByte().toLong().toCPointer()
        is Char -> this.code.toLong().toCPointer()
        is Byte -> this.toLong().toCPointer()
        is Short -> this.toLong().toCPointer()
        is Int -> this.toLong().toCPointer()
        is Long -> this.toCPointer()
        is UByte -> this.toLong().toCPointer()
        is UShort -> this.toLong().toCPointer()
        is UInt -> this.toLong().toCPointer()
        is ULong -> this.toLong().toCPointer()
        is Float -> this.toBits().toLong().toCPointer()
        is Double -> this.toBits().toCPointer()
        is String -> {
            val bytes = this.cstr.getBytes()
            val data = nativeHeap.allocArray<ByteVar>(bytes.size)
            for (i in 0 until bytes.size) {
                data[i] = bytes[i]
            }
            data
        }
        else -> {
            StableRef.create(this).asCPointer()
        }
    }
}
