#ifndef LIB_FLUTTER_EMBEDDER_METHOD_CALL_H
#define LIB_FLUTTER_EMBEDDER_METHOD_CALL_H

#include <memory>
#include <string>
#include <utility>

#include "encodable.h"
#include "export.h"
#include "logger.h"
#include "method-codec-interface.h"

typedef struct _FlutterPlatformMessageResponseHandle FlutterResponseHandleFwd;

class EMBEDDER_EXPORT MethodCall final
{
private:
    friend class MethodChannel;
    MethodCall(std::shared_ptr<IMethodCodec> codec,
               const std::string &method,
               const Encodable &args,
               const FlutterResponseHandleFwd *responseHandle);

    template<typename T>
    constexpr void validateArgumentType() const
    {
        /* clang-format off */
        static_assert(std::is_same_v<T, Encodable::Null> ||
                      std::is_same_v<T, Encodable::Boolean> ||
                      std::is_same_v<T, Encodable::Int> ||
                      std::is_same_v<T, Encodable::Float> ||
                      std::is_same_v<T, Encodable::String> ||
                      std::is_same_v<T, Encodable::Uint8List> ||
                      std::is_same_v<T, Encodable::Int32List> ||
                      std::is_same_v<T, Encodable::Int64List> ||
                      std::is_same_v<T, Encodable::Float32List> ||
                      std::is_same_v<T, Encodable::Float64List> ||
                      std::is_same_v<T, Encodable::List> ||
                      std::is_same_v<T, Encodable::Map>,
                      "attempts to get the argument with unsupported type from method call");
        /* clang-format on */
    }

public:
    const std::string &GetMethod() const;
    const Encodable &GetArguments() const;

    template<typename T>
    const T &GetArgument(const Encodable &key) const
    {
        validateArgumentType<T>();

        if (!m_args.IsMap())
            logcrit << "attempts to get the argument by key from method call, "
                       "but method call arguments is not "
                    << Encodable::Type::Map << std::endl;

        if constexpr (std::is_same_v<T, Encodable::Null>)
            return m_args[key].GetNull();

        if constexpr (std::is_same_v<T, Encodable::Boolean>)
            return m_args[key].GetBoolean();

        if constexpr (std::is_same_v<T, Encodable::Int>)
            return m_args[key].GetInt();

        if constexpr (std::is_same_v<T, Encodable::Float>)
            return m_args[key].GetFloat();

        if constexpr (std::is_same_v<T, Encodable::String>)
            return m_args[key].GetString();

        if constexpr (std::is_same_v<T, Encodable::Uint8List>)
            return m_args[key].GetUint8List();

        if constexpr (std::is_same_v<T, Encodable::Int32List>)
            return m_args[key].GetInt32List();

        if constexpr (std::is_same_v<T, Encodable::Int64List>)
            return m_args[key].GetInt64List();

        if constexpr (std::is_same_v<T, Encodable::Float32List>)
            return m_args[key].GetFloat32List();

        if constexpr (std::is_same_v<T, Encodable::Float64List>)
            return m_args[key].GetFloat64List();

        if constexpr (std::is_same_v<T, Encodable::List>)
            return m_args[key].GetList();

        if constexpr (std::is_same_v<T, Encodable::Map>)
            return m_args[key].GetMap();

        logcrit << "unexpected type found" << std::endl;

        /*
            Unreachable, since either `static_assert` will fire, or a value
            will be returned, or the program will exit with `logcrit`.

            This is done so that there is no warning about the return value.
        */
        static const T dummy{};
        return dummy;
    }

    template<typename T>
    T &GetArgument(const Encodable &key)
    {
        return const_cast<T &>(std::as_const(*this).GetArgument<T>(key));
    }

    template<typename T>
    T GetArgument(const Encodable &key)
    {
        return GetArgument<T>(key);
    }

    template<typename T>
    T GetArgument(const Encodable &key, const T &fallback) const
    {
        validateArgumentType<T>();

        if (!m_args.HasKey(key))
            return fallback;

        return GetArgument<T>(key);
    }

public:
    void SendSuccessResponse(const Encodable &result) const;
    void SendErrorResponse(const std::string &code,
                           const std::string &message,
                           const Encodable &details) const;

private:
    std::shared_ptr<IMethodCodec> m_codec;
    std::string m_method;
    Encodable m_args;
    const FlutterResponseHandleFwd *m_responseHandle;
};

#endif /* LIB_FLUTTER_EMBEDDER_METHOD_CALL_H */
