#ifndef LIB_FLUTTER_EMBEDDER_ENCODABLE_H
#define LIB_FLUTTER_EMBEDDER_ENCODABLE_H

#include <cstddef>
#include <map>
#include <string>
#include <variant>
#include <vector>

#include "export.h"

class Encodable;

typedef std::variant<std::monostate,
                     bool,
                     int64_t,
                     double,
                     std::string,
                     std::vector<uint8_t>,
                     std::vector<int32_t>,
                     std::vector<int64_t>,
                     std::vector<float>,
                     std::vector<double>,
                     std::vector<Encodable>,
                     std::map<Encodable, Encodable>>
    EncodableVariant;

class EMBEDDER_EXPORT Encodable final : public EncodableVariant
{
public:
    typedef nullptr_t Null;
    typedef bool Boolean;
    typedef int64_t Int;
    typedef double Float;
    typedef std::string String;
    typedef std::vector<uint8_t> Uint8List;
    typedef std::vector<int32_t> Int32List;
    typedef std::vector<int64_t> Int64List;
    typedef std::vector<float> Float32List;
    typedef std::vector<double> Float64List;
    typedef std::vector<Encodable> List;
    typedef std::map<Encodable, Encodable> Map;

    enum class Type {
        Null = 0,
        Boolean,
        Int,
        Float,
        String,
        Uint8List,
        Int32List,
        Int64List,
        Float64List,
        List,
        Map,
        Float32List,
    };

public:
    Encodable();
    Encodable(Null null);
    Encodable(Boolean boolean);
    Encodable(Float floating);
    Encodable(const String &string);
    Encodable(const char *string);
    Encodable(const Uint8List &u8array);
    Encodable(const Int32List &i32array);
    Encodable(const Int64List &i64array);
    Encodable(const Float32List &f32array);
    Encodable(const Float64List &f64array);
    Encodable(const List &list);
    Encodable(const Map &map);

    template<typename Integral>
    Encodable(Integral integral)
        : EncodableVariant(static_cast<Int>(integral))
    {
        static_assert(std::is_integral_v<Integral>,
                      "constructor called for integral type,"
                      "but Integral is not integral type");
    }

public:
    Null &GetNull();
    Boolean &GetBoolean();
    Int &GetInt();
    Float &GetFloat();
    String &GetString();
    Uint8List &GetUint8List();
    Int32List &GetInt32List();
    Int64List &GetInt64List();
    Float32List &GetFloat32List();
    Float64List &GetFloat64List();
    List &GetList();
    Map &GetMap();

    const Null &GetNull() const;
    const Boolean &GetBoolean() const;
    const Int &GetInt() const;
    const Float &GetFloat() const;
    const String &GetString() const;
    const Uint8List &GetUint8List() const;
    const Int32List &GetInt32List() const;
    const Int64List &GetInt64List() const;
    const Float32List &GetFloat32List() const;
    const Float64List &GetFloat64List() const;
    const List &GetList() const;
    const Map &GetMap() const;

    bool IsNull() const;
    bool IsBoolean() const;
    bool IsInt() const;
    bool IsFloat() const;
    bool IsString() const;
    bool IsUint8List() const;
    bool IsInt32List() const;
    bool IsInt64List() const;
    bool IsFloat32List() const;
    bool IsFloat64List() const;
    bool IsList() const;
    bool IsMap() const;

    Type GetType() const;

    bool HasKey(const Encodable &key) const;
    size_t Size() const;

    const Encodable &operator[](size_t index) const;
    Encodable &operator[](size_t index);

    const Encodable &operator[](const Encodable &key) const;
    Encodable &operator[](const Encodable &key);
};

EMBEDDER_EXPORT
std::ostream &operator<<(std::ostream &os, Encodable::Type type);

EMBEDDER_EXPORT
std::ostream &operator<<(std::ostream &os, const Encodable &encodable);

#endif /* LIB_FLUTTER_EMBEDDER_ENCODABLE_H */
