// SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause
import 'package:flutter/material.dart';

import 'theme/radius.dart';

/// Common layout-widget for list work with [ListItemData]
class ListItem extends StatelessWidget {
  const ListItem(
    this.title,
    this.description,
    this.color,
    this.value, {
    super.key,
    this.widthData,
    this.buttonText = '',
    this.buttonHide = true,
    this.buttonOnPressed,
  });

  final String title;
  final String? description;
  final Color color;
  final dynamic value;
  final double? widthData;
  final String buttonText;
  final bool buttonHide;
  final VoidCallback? buttonOnPressed;

  @override
  Widget build(BuildContext context) {
    final isEmptyBody = value.toString() == 'SizedBox.shrink';
    return Card(
      surfaceTintColor: color,
      child: Padding(
          padding: const EdgeInsets.all(16),
          child: widthData == null
              ? Column(
                  children: [
                    _dataInfo(),
                    if (!isEmptyBody) const SizedBox(height: 14),
                    if (!isEmptyBody) _dataValue(),
                    _button(),
                  ],
                )
              : Column(
                  children: [
                    IntrinsicHeight(
                      child: Row(
                        crossAxisAlignment: CrossAxisAlignment.stretch,
                        mainAxisAlignment: MainAxisAlignment.spaceBetween,
                        children: [
                          Flexible(
                            child: _dataInfo(),
                          ),
                          if (!isEmptyBody) const SizedBox(width: 16),
                          if (!isEmptyBody) _dataValue()
                        ],
                      ),
                    ),
                    _button(),
                  ],
                )),
    );
  }

  Widget _dataInfo() {
    return SizedBox(
      width: double.infinity,
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(
            title,
            style: const TextStyle(
              fontSize: 14,
              color: Colors.black,
              fontWeight: FontWeight.bold,
            ),
          ),
          if (description != null && description!.isNotEmpty)
            Padding(
              padding: const EdgeInsets.only(top: 6),
              child: Text(
                description!,
                style: const TextStyle(
                  fontSize: 12,
                  color: Colors.grey,
                ),
              ),
            )
        ],
      ),
    );
  }

  Widget _dataValue() {
    return Container(
      width: widthData,
      padding: EdgeInsets.all(widthData == null ? 8 : 6),
      alignment: Alignment.center,
      decoration: BoxDecoration(
        borderRadius: InternalRadius.medium,
        color: color,
      ),
      child: value is Widget
          ? value
          : Text(
              value == null ? 'NULL' : value.toString(),
              style: const TextStyle(
                fontSize: 14,
                color: Colors.white,
                fontWeight: FontWeight.bold,
              ),
            ),
    );
  }

  Widget _button() {
    if (buttonHide) {
      return const SizedBox.shrink();
    }
    return Column(
      children: [
        const SizedBox(height: 16),
        SizedBox(
          width: double.infinity,
          child: TextButton(
            style: TextButton.styleFrom(
              backgroundColor: color,
              disabledBackgroundColor: Colors.black45,
              foregroundColor: Colors.white,
              disabledForegroundColor: Colors.white.withOpacity(0.7),
              padding: const EdgeInsets.symmetric(vertical: 20, horizontal: 24),
              shape: RoundedRectangleBorder(
                borderRadius: InternalRadius.medium,
              ),
            ),
            onPressed: buttonOnPressed,
            child: Text(buttonText.toUpperCase()),
          ),
        )
      ],
    );
  }
}
