/* 
 * SPDX-FileCopyrightText: Copyright 2021 Sony Group Corporation,
 * SPDX-FileCopyrightText: Copyright 2023 Alexander Syrykh,
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef FLUTTER_PLUGIN_AUDIOPLAYERS_AURORA_GST_AUDIO_PLAYER_H_
#define FLUTTER_PLUGIN_AUDIOPLAYERS_AURORA_GST_AUDIO_PLAYER_H_

#include <gst/gst.h>

#include <atomic>
#include <functional>
#include <memory>
#include <mutex>
#include <string>
#include <thread>
#include <vector>

#include <audioplayers_aurora/types/audio_player_stream_handler.h>

class GstAudioPlayer {
public:
    GstAudioPlayer(const std::string& player_id, std::unique_ptr<AudioPlayerStreamHandler> handler);
    ~GstAudioPlayer();

    static void GstLibraryLoad();
    static void GstLibraryUnload();

    void Resume();
    void Play();
    void Pause();
    void Stop();
    void Seek(int64_t position);
    void SetSourceUrl(std::string url);
    void SetVolume(double volume);
    void SetBalance(double balance);
    void SetPlaybackRate(double playback_rate);
    void SetLooping(bool is_looping);
    int64_t GetDuration();
    int64_t GetCurrentPosition();
    void Release();
    void Dispose();

private:
    struct GstAudioElements {
        GstElement* playbin = nullptr;
        GstBus* bus = nullptr;
        GstElement* source = nullptr;
        GstElement* panorama = nullptr;
        GstElement* audiobin = nullptr;
        GstElement* audiosink = nullptr;
        GstPad* panoramasinkpad = nullptr;
    };

    static GstBusSyncReply HandleGstMessage(GstBus*, GstMessage* message, gpointer user_data);
    static void SourceSetup(GstElement*, GstElement* source, GstElement**);
    bool CreatePipeline();
    std::string ParseUri(const std::string& uri);
    void SendPositionPeriodically();

    GstAudioElements gst_;
    const std::string player_id_;
    std::string url_;
    bool is_initialized_ = false;
    bool is_looping_ = false;
    bool is_completed_ = false;
    double volume_ = 1.0;
    double playback_rate_ = 1.0;
    std::mutex mutex_;
    std::thread position_sender;
    std::atomic<bool> is_playing_ = false;
    std::atomic<bool> need_to_send_position_event_ = true;
    std::unique_ptr<AudioPlayerStreamHandler> stream_handler_;
};

#endif  // FLUTTER_PLUGIN_AUDIOPLAYERS_AURORA_GST_AUDIO_PLAYER_H_
