/* 
 * SPDX-FileCopyrightText: Copyright 2021 Sony Group Corporation,
 * SPDX-FileCopyrightText: Copyright 2023 Alexander Syrykh,
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <map>
#include <memory>
#include <optional>
#include <string>
#include <variant>

#include <flutter/basic_message_channel.h>
#include <flutter/encodable_value.h>
#include <flutter/event_channel.h>
#include <flutter/event_sink.h>
#include <flutter/event_stream_handler_functions.h>
#include <flutter/method_channel.h>
#include <flutter/plugin_registrar.h>
#include <flutter/standard_message_codec.h>
#include <flutter/standard_method_codec.h>

#include <audioplayers_aurora/audio_player_stream_handler_impl.h>
#include <audioplayers_aurora/audioplayers_aurora_plugin.h>
#include <audioplayers_aurora/gst_audio_player.h>

constexpr auto kInvalidArgument = "Invalid argument";
constexpr auto kAudioDurationEvent = "audio.onDuration";
constexpr auto kAudioPositionEvent = "audio.onCurrentPosition";
constexpr auto kAudioPreparedEvent = "audio.onPrepared";
constexpr auto kAudioSeekCompleteEvent = "audio.onSeekComplete";
constexpr auto kAudioCompleteEvent = "audio.onComplete";
constexpr auto kAudioLogEvent = "audio.onLog";

template <typename T>
std::optional<T> GetValueFromEncodableMap(const flutter::EncodableMap* encodable_map, const char* key) {
    auto it = encodable_map->find(flutter::EncodableValue(key));
    if (it == encodable_map->end()) {
        return {};
    }

    auto value = it->second;
    if (value.IsNull()) {
        return {};
    }

    if (!std::holds_alternative<T>(value)) {
        return {};
    }

    return std::get<T>(value);
}

void AudioplayersAuroraPlugin::RegisterWithRegistrar(flutter::PluginRegistrar* registrar) {
    auto plugin = std::unique_ptr<AudioplayersAuroraPlugin>(new AudioplayersAuroraPlugin(registrar));
    plugin->InitializeHandlers();
    registrar->AddPlugin(std::move(plugin));
}

void AudioplayersAuroraPlugin::InitializeHandlers() {
    auto channel = std::make_unique<flutter::MethodChannel<flutter::EncodableValue>>(
        registrar_->messenger(), "xyz.luan/audioplayers", &flutter::StandardMethodCodec::GetInstance());

    channel->SetMethodCallHandler([this](const auto& call, auto result) { HandleMethodCall(call, std::move(result)); });
}

AudioplayersAuroraPlugin::AudioplayersAuroraPlugin(flutter::PluginRegistrar* registrar) : registrar_(registrar) {
    GstAudioPlayer::GstLibraryLoad();
}

void AudioplayersAuroraPlugin::HandleMethodCall(
    const flutter::MethodCall<flutter::EncodableValue>& method_call,
    std::unique_ptr<flutter::MethodResult<flutter::EncodableValue>> result) {
    const auto arguments = std::get_if<flutter::EncodableMap>(method_call.arguments());
    if (!arguments) {
        result->Error(kInvalidArgument, "No arguments provided.");
        return;
    }

    const auto player_id = GetValueFromEncodableMap<std::string>(arguments, "playerId");
    if (!player_id) {
        result->Error(kInvalidArgument, "No playerId provided.");
        return;
    }

    const auto method_name = method_call.method_name();
    if (method_name == "create") {
        CreateAudioPlayer(*player_id);
        result->Success();
        return;
    }

    const auto player = GetAudioPlayer(*player_id);
    if (!player) {
        result->Error(kInvalidArgument, "No AudioPlayer with id " + *player_id + " is exist.");
        return;
    }

    if (method_name == "resume") {
        player->Resume();
        result->Success();
        return;
    }

    if (method_name == "pause") {
        player->Pause();
        result->Success();
        return;
    }

    if (method_name == "stop") {
        player->Stop();
        result->Success();
        return;
    }

    if (method_name == "release") {
        player->Release();
        result->Success();
        return;
    }

    if (method_name == "seek") {
        const auto position = GetValueFromEncodableMap<int32_t>(arguments, "position");
        if (!position) {
            result->Error(kInvalidArgument, "No \"position\" provided");
            return;
        }

        player->Seek(*position);
        result->Success();
        return;
    }

    if (method_name == "setVolume") {
        const auto volume = GetValueFromEncodableMap<double>(arguments, "volume");
        if (!volume) {
            result->Error(kInvalidArgument, "No \"volume\" provided");
            return;
        }

        player->SetVolume(*volume);
        result->Success();
        return;
    }

    if (method_name == "setSourceUrl") {
        auto url = GetValueFromEncodableMap<std::string>(arguments, "url").value_or(std::string{});
        if (url.empty()) {
            result->Error(kInvalidArgument, "No \"URL\" provided");
            return;
        }

        const auto is_local = GetValueFromEncodableMap<bool>(arguments, "isLocal");
        if (is_local.value_or(false)) {
            url = std::string("file://") + url;
        }

        player->SetSourceUrl(url);
        result->Success();
        return;
    }

    if (method_name == "setPlaybackRate") {
        const auto rate = GetValueFromEncodableMap<double>(arguments, "playbackRate");
        if (!rate) {
            result->Error(kInvalidArgument, "No \"rate\" provided");
            return;
        }

        player->SetPlaybackRate(*rate);
        result->Success();
        return;
    }

    if (method_name == "setReleaseMode") {
        const auto release_mode = GetValueFromEncodableMap<std::string>(arguments, "releaseMode");
        if (!release_mode) {
            result->Error(kInvalidArgument, "No \"release mode\" provided");
            return;
        }

        bool looping = release_mode->find("loop") != std::string::npos;
        player->SetLooping(looping);
        result->Success();
        return;
    }

    if (method_name == "getDuration") {
        int64_t duration = player->GetDuration();
        if (duration >= 0) {
            result->Success(flutter::EncodableValue(duration));
        } else {
            result->Success();
        }

        return;
    }

    if (method_name == "getCurrentPosition") {
        int64_t position = player->GetCurrentPosition();
        if (position >= 0) {
            result->Success(flutter::EncodableValue(position));
        } else {
            result->Success();
        }

        return;
    }

    if (method_name == "setBalance") {
        const auto balance = GetValueFromEncodableMap<double>(arguments, "balance");
        if (!balance) {
            result->Error(kInvalidArgument, "No \"balance\" provided");
            return;
        }

        player->SetBalance(*balance);
        result->Success();
        return;
    }

    if (method_name == "dispose") {
        player->Dispose();
        audio_players_.erase(*player_id);
        event_sinks_.erase(*player_id);
        result->Success();
        return;
    }

    result->NotImplemented();
}

GstAudioPlayer* AudioplayersAuroraPlugin::GetAudioPlayer(const std::string& player_id) {
    auto it = audio_players_.find(player_id);
    if (it != audio_players_.end()) {
        return it->second.get();
    }
    return nullptr;
}

void AudioplayersAuroraPlugin::CreateAudioPlayer(const std::string& player_id) {
    InitializeEventChannel(player_id);
    InitializeAudioPlayer(player_id);
}

void AudioplayersAuroraPlugin::InitializeEventChannel(const std::string& player_id) {
    auto event_channel = std::make_unique<flutter::EventChannel<flutter::EncodableValue>>(
        registrar_->messenger(), "xyz.luan/audioplayers/events/" + player_id,
        &flutter::StandardMethodCodec::GetInstance());

    auto const on_listen = [this, id = player_id](const flutter::EncodableValue*,
                                                  std::unique_ptr<flutter::EventSink<flutter::EncodableValue>>&& events)
        -> std::unique_ptr<flutter::StreamHandlerError<flutter::EncodableValue>> {
        this->event_sinks_[id] = std::move(events);
        return nullptr;
    };

    auto const on_cancel =
        [](const flutter::EncodableValue*) -> std::unique_ptr<flutter::StreamHandlerError<flutter::EncodableValue>> {
        return nullptr;
    };

    auto event_channel_handler =
        std::make_unique<flutter::StreamHandlerFunctions<flutter::EncodableValue>>(on_listen, on_cancel);

    event_channel->SetStreamHandler(std::move(event_channel_handler));
}

void AudioplayersAuroraPlugin::InitializeAudioPlayer(const std::string& player_id) {
    auto const on_notify_prepared = [this](const std::string& player_id, bool is_prepared) {
        flutter::EncodableMap map = {{flutter::EncodableValue("event"), flutter::EncodableValue(kAudioPreparedEvent)},
                                     {flutter::EncodableValue("value"), flutter::EncodableValue(is_prepared)}};
        event_sinks_[player_id]->Success(flutter::EncodableValue(map));
    };

    auto const on_notify_duration = [this](const std::string& player_id, int32_t duration) {
        flutter::EncodableMap map = {{flutter::EncodableValue("event"), flutter::EncodableValue(kAudioDurationEvent)},
                                     {flutter::EncodableValue("value"), flutter::EncodableValue(duration)}};
        event_sinks_[player_id]->Success(flutter::EncodableValue(map));
    };

    auto const on_notify_position = [this](const std::string& player_id, int32_t position) {
        flutter::EncodableMap map = {{flutter::EncodableValue("event"), flutter::EncodableValue(kAudioPositionEvent)},
                                     {flutter::EncodableValue("value"), flutter::EncodableValue(position)}};
        event_sinks_[player_id]->Success(flutter::EncodableValue(map));
    };

    auto const on_notify_seek_completed = [this](const std::string& player_id) {
        flutter::EncodableMap map = {
            {flutter::EncodableValue("event"), flutter::EncodableValue(kAudioSeekCompleteEvent)}};
        event_sinks_[player_id]->Success(flutter::EncodableValue(map));
    };

    auto const on_notify_play_completed = [this](const std::string& player_id) {
        flutter::EncodableMap map = {{flutter::EncodableValue("event"), flutter::EncodableValue(kAudioCompleteEvent)}};
        event_sinks_[player_id]->Success(flutter::EncodableValue(map));
    };

    auto const on_notify_log = [this](const std::string& player_id, const std::string& message) {
        flutter::EncodableMap map = {{flutter::EncodableValue("event"), flutter::EncodableValue(kAudioLogEvent)},
                                     {flutter::EncodableValue("value"), flutter::EncodableValue(message)}};
        event_sinks_[player_id]->Success(flutter::EncodableValue(map));
    };

    auto player_handler = std::make_unique<AudioPlayerStreamHandlerImpl>(on_notify_prepared, on_notify_duration,
                                                                         on_notify_position, on_notify_seek_completed,
                                                                         on_notify_play_completed, on_notify_log);

    auto player = std::make_unique<GstAudioPlayer>(player_id, std::move(player_handler));
    audio_players_[player_id] = std::move(player);
}
