// SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include <QStandardPaths>
#include <QSettings>
#include <QtQml>

#include "applicationsettings.h"

#define DOC_SETTINGS_FILE_PATH QStringLiteral("%1/%2")\
    .arg(QStandardPaths::writableLocation(QStandardPaths::AppConfigLocation))\
    .arg("docSettings.conf")

#define APP_SETTINGS_FILE_PATH QStringLiteral("%1/%2")\
    .arg(QStandardPaths::writableLocation(QStandardPaths::AppConfigLocation))\
    .arg("appSettings.conf")

static const QString s_verticalScrollSettingsName = QStringLiteral("verticalScroll");
static const QString s_notesPaintSettingsName = QStringLiteral("notesPaint");
static const QString s_documentSectionName = QStringLiteral("Documents");
static const QString s_documentZoom = QStringLiteral("zoom");
static const QString s_documentPosition = QStringLiteral("position");
static const QString s_documentLastOpen = QStringLiteral("lastOpen");

static const QString s_appConfigSection = QStringLiteral("AppConfig");
static const QString s_welcomeFileCopied = QStringLiteral("welcomeFileCopied");

static DocumentSettings *s_instance;

DocumentSettings::DocumentSettings(QObject *parent) : QObject(parent)
{
    m_settings.reset(new QSettings(DOC_SETTINGS_FILE_PATH, QSettings::Format::NativeFormat, this));
}

bool DocumentSettings::notesPaint(const QString &name) const
{
    m_settings->beginGroup(s_documentSectionName);
    auto notesPaint = m_settings->value(QString("%1/%2").arg(name).arg(s_notesPaintSettingsName), true).toBool();
    m_settings->endGroup();

    return notesPaint;
}

bool DocumentSettings::verticalScroll(const QString &name) const
{
    m_settings->beginGroup(s_documentSectionName);
    auto verticalScroll = m_settings->value(QString("%1/%2").arg(name).arg(s_verticalScrollSettingsName), true).toBool();
    m_settings->endGroup();

    return verticalScroll;
}

void DocumentSettings::registerSingleton(const char *uri)
{
    qmlRegisterSingletonType<DocumentSettings>(uri, 1, 0, "DocumentSettings", DocumentSettings::singletonProvider);
}

QObject *DocumentSettings::singletonProvider(QQmlEngine *engine, QJSEngine *)
{
    if (!s_instance)
        s_instance = new DocumentSettings(engine);

    return s_instance;
}

void DocumentSettings::setNotesPaint(const QString &name, bool notesPaint)
{
    m_settings->beginGroup(s_documentSectionName);
    m_settings->setValue(QString("%1/%2").arg(name).arg(s_notesPaintSettingsName), qVariantFromValue(notesPaint));
    m_settings->endGroup();
}

void DocumentSettings::setVerticalScroll(const QString &name, bool verticalScroll)
{
    m_settings->beginGroup(s_documentSectionName);
    m_settings->setValue(QString("%1/%2").arg(name).arg(s_verticalScrollSettingsName), qVariantFromValue(verticalScroll));
    m_settings->endGroup();
}

void DocumentSettings::setDocumentZoomAndPosition(const QString &name, qreal zoom, const QPointF &position)
{
    m_settings->beginGroup(s_documentSectionName);
    m_settings->setValue(QString("%1/%2").arg(name).arg(s_documentZoom), qVariantFromValue(float(zoom)));
    m_settings->setValue(QString("%1/%2").arg(name).arg(s_documentPosition), qVariantFromValue(position));
    m_settings->endGroup();
}

float DocumentSettings::getDocumentZoom(const QString &name) const
{
    m_settings->beginGroup(s_documentSectionName);
    auto zoom = m_settings->value(QString("%1/%2").arg(name).arg(s_documentZoom), 1.0f).toFloat();
    m_settings->endGroup();

    return zoom;
}

QPointF DocumentSettings::getDocumentPosition(const QString &name) const
{
    m_settings->beginGroup(s_documentSectionName);
    auto position = m_settings->value(QString("%1/%2").arg(name).arg(s_documentPosition), QPointF(0.0, 0.0)).toPointF();
    m_settings->endGroup();

    return position;
}

void DocumentSettings::setDocumentLastOpen(const QString &name)
{
    m_settings->beginGroup(s_documentSectionName);
    m_settings->setValue(QString("%1/%2").arg(name).arg(s_documentLastOpen), qVariantFromValue(QDateTime::currentDateTimeUtc()));
    m_settings->endGroup();
}

ApplicationSettings &ApplicationSettings::singletonProvider()
{
    static ApplicationSettings provider;
    return provider;
}

ApplicationSettings::ApplicationSettings(QObject *parent)
    : QObject(parent)
    , m_settings(new QSettings(APP_SETTINGS_FILE_PATH, QSettings::Format::NativeFormat, this))
{
    if (!m_settings->contains(QString("%1/%2").arg(s_appConfigSection, s_welcomeFileCopied))) {
        setWelcomeFileCopied(false);
    }
}

void ApplicationSettings::setWelcomeFileCopied(bool copied)
{
    m_settings->beginGroup(s_appConfigSection);
    m_settings->setValue(QString("%1").arg(s_welcomeFileCopied), qVariantFromValue(copied));
    m_settings->endGroup();
}

bool ApplicationSettings::welcomeFileCopied() const
{
    m_settings->beginGroup(s_appConfigSection);
    const auto copied = m_settings->value(QString("%1").arg(s_welcomeFileCopied), true).toBool();
    m_settings->endGroup();

    return copied;
}
