// SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Share 1.0
import Aurora.Controls 1.0
import ru.auroraos.amberpdf 1.0
import ru.auroraos.PdfViewer 1.0
import Sailfish.Pickers 1.0
import "../components"

Page {
    id: root

    property string pdfPath
    property int _maxSize: Math.max(Screen.width, Screen.height)
    property bool fullscreenMode: false
    property bool noteCreatingMode: false
    property bool isNotePointChosen: false
    readonly property var statesNames: ["correctWork", "loadingDocument", "badFile", "fileNotFound", "noteCreating"]

    function checkCurrentFile() {
        fileInfo.refresh();
        if (fileInfo.isExist())
            return;
        state = root.statesNames[3];
    }

    function splitDocumentName() {
        return fileInfo.fileName + fileInfo.lastModified + fileInfo.size

    }

    function loadViewPosition() {
        var zoom = DocumentSettings.getDocumentZoom(splitDocumentName())
        var position = DocumentSettings.getDocumentPosition(splitDocumentName())

        pdfPagesView.zoom = zoom
        pdfPagesView.contentX = position.x
        pdfPagesView.contentY = position.y
        pdfPagesView.correctPosition()
    }

    onStatusChanged: {
        if (pageStack.currentPage.objectName === objectName) {
            if (status === PageStatus.Active && state === statesNames[1]) {
                pdfPagesView.pdfPath = pdfPath;
            }
        }
    }

    allowedOrientations: Orientation.All
    objectName: "contentPage"
    state: root.statesNames[1]

    TextArea {
        id: errorText

        width: parent.width
        visible: root.state !== root.statesNames[0]
        color: Theme.highlightColor
        readOnly: true
        horizontalAlignment: TextEdit.AlignHCenter
        font.pixelSize: Theme.fontSizeExtraLarge
        anchors {
            verticalCenter: parent.verticalCenter
            horizontalCenter: parent.horizontalCenter
        }

        background: Rectangle {
            color: "transparent"
            border.color: "transparent"
        }
    }

    PdfView {
        id: pdfPagesView

        property real previousStatus: PdfDocument.Null

        documentProvider: pdfiumProvider
        clip: true
        width: parent.width
        catchBound: appBar.height
        annotationsPaint: true
        notesPaint: true
        anchors.horizontalCenter: parent.horizontalCenter

        onClickedUrl: Qt.openUrlExternally(url)
        onNoteActivate: pageStack.push(Qt.resolvedUrl("NotePage.qml"), {
                                           noteText: noteText,
                                           author: author,
                                           pageNumber: pageIndex + 1,
                                           color: color,
                                           noteId: annotationId,
                                           pdfPagesView: pdfPagesView,
                                           noteType: noteType,
                                           editable: pdfPagesView.isPageInPainted(pageIndex)
                                       })

        onStatusChanged: {
            if (previousStatus === status)
                return
            switch (previousStatus) {
            case PdfDocument.Null:
                root.state = (status === PdfDocument.Loading ? root.statesNames[1] : root.statesNames[2])
                break
            case PdfDocument.Loading:
                loadViewPosition()
                timerLoadViewPosition.start()

                timer.start()
                root.state = (status === PdfDocument.Ready ? root.statesNames[0] : root.statesNames[2])
                break
            case PdfDocument.Ready:
                root.state = root.statesNames[3]
                previousStatus = PdfDocument.Error
                break
            }

            if (previousStatus !== PdfDocument.Error)
                previousStatus = status
        }

        onHolding: {
            if (!noteCreatingMode)
                return
            if (isNotePointChosen)
                return

            isNotePointChosen = true
            var createNotePage = pageStack.push(Qt.resolvedUrl("CreateNotePage.qml"))
            createNotePage.onCreateNote.connect(function (text, color, noteType) {
                pdfPagesView.addAnnotationAtPoint(screenCoordinates, "defaultuser", text, color, noteType)
                root.state = root.statesNames[0]
            });
            isNotePointChosen = false
        }

        Binding {
            target: pdfPagesView
            property: "height"
            value: appBar.opened
                   ? root.height - appBar.height - toolBar.height
                   : root.height
        }
        Binding {
            target: pdfPagesView
            property: "y"
            value: appBar.opened
                   ? appBar.height
                   : 0
        }

        Behavior on height {
            enabled: !timer.running
            NumberAnimation {
                duration: root.state === root.statesNames[0] ? toolBar.animationDuration : 0
                easing.type: Easing.InOutQuad
            }
        }

        Behavior on height {
            enabled: !timer.running
            NumberAnimation {
                duration: root.state === root.statesNames[0] ? toolBar.animationDuration : 0
                easing.type: Easing.InOutQuad
            }
        }

        Behavior on y {
            enabled: !timer.running
            NumberAnimation {
                duration: root.state === root.statesNames[0] ? toolBar.animationDuration : 0
                easing.type: Easing.InOutQuad
            }
        }

        Timer {
            id: timer

            running: false
            repeat: false
            interval: 1000
        }

        Timer {
            id: timerLoadViewPosition

            onTriggered: loadViewPosition()

            interval: 150
        }
    }

    ContentHeader {
        id: appBar

        visibleButtons: root.state === root.statesNames[0]
        fullscreenMode: root.fullscreenMode
        noteCreatingMode: root.state === root.statesNames[4]

        title: fileInfo.fileName

        onOpenDetails: pageStack.push(Qt.resolvedUrl("DetailsPage.qml"), {
                                          "pageCount":
                                            root.state === root.statesNames[1]
                                            ? qsTr("Loading")
                                            : pdfPagesView.count,
                                          "fileInfo": fileInfo
                                      })

        onFullscreenClicked: {
            root.fullscreenMode = true
        }
        onCancelNoteCreatingClicked: {
            root.state = root.statesNames[0]
        }
    }

    ToolBar {
        id: toolBar

        anchors.bottom: parent.bottom

        width: parent.width
        opened: !root.fullscreenMode

        pageIndex: pdfPagesView.currentIndex
        pagesCount: pdfPagesView.count

        onOpenPopupMenu: popupMenu.openPopup(item)
        onOpenPopupNotes: popupMenuNotes.openPopup(item)

        onOpenNavigationPage: {
           var navigationPage = pageStack.push(Qt.resolvedUrl("NavigationPage.qml"), {
                                                    count: pdfPagesView.count,
                                                    bookmarks: pdfPagesView.bookmarksModel
                                                })
           navigationPage.pageSelected.connect(function(page) { pdfPagesView.goToPage(page - 1) })
           navigationPage.sectionSelected.connect(function(page) {
               pdfPagesView.goToPage(page)
           })
        }
    }

    Button {
        onClicked: {
            fullscreenMode = false
        }

        anchors {
            top: parent.top
            topMargin: Theme.paddingMedium
            right: parent.right
            rightMargin: Theme.paddingMedium
        }

        opacity: root.fullscreenMode ? 1.0 : 0.0
        visible: opacity
        backgroundColor: Theme.rgba(Theme.highlightBackgroundColor, Theme.opacityOverlay)
        icon.source: "image://theme/icon-splus-collapse"

        Behavior on opacity {
            FadeAnimation {}
        }
    }

    PdfDocument {
        id: pdfiumProvider

        objectName: "pdfDocument"
    }

    PopupMenu {
        id: popupMenu

        property var delegate

        function openPopup(d) {
            d.selected = true
            delegate = d
            open(d)
        }

        onOpenedChanged: delegate.selected = opened

        preferredWidth: applicationWindow.preferredWidthPopup

        PopupMenuCheckableItem {
            onCheckedChanged: {
                pdfPagesView.orientation = (pdfPagesView.orientation === Qt.Vertical ? Qt.Horizontal : Qt.Vertical);
                DocumentSettings.setVerticalScroll(splitDocumentName(), pdfPagesView.orientation === Qt.Vertical);
            }

            checked: pdfPagesView.orientation === Qt.Horizontal
            text: qsTr("Scroll pages horizontally")
        }

        PopupMenuDividerItem {}

        PopupMenuItem {
            onClicked: {
                shareAction.resources = [Qt.resolvedUrl(fileInfo.path)]
                shareAction.trigger()
            }

            icon.source: "image://theme/icon-splus-share"
            text: qsTr("Share")
        }

        PopupMenuItem {
            onClicked: {
                var searchPage = pageStack.push(Qt.resolvedUrl(
                                                    "SearchPhrasePage.qml"), {
                                                    "pdfPagesView": pdfPagesView
                                                })
                searchPage.pageSelected.connect(function (page) {
                    pdfPagesView.goToPage(page)
                })
            }

            icon.source: "image://theme/icon-m-search"
            text: qsTr("Search by word")
        }

        PopupMenuItem {
            onClicked: {
                pageStack.push(folderPickerDialog)
            }

            icon.source: "image://theme/icon-m-save"
            text: qsTr("Save document")
        }
    }

    Component {
        id: folderPickerDialog

        FolderPickerDialog {
            title: qsTr("Save into")
            onAccepted: pdfPagesView.saveDocumentAs(selectedPath + "/" + fileInfo.fileName)
        }
    }

    PopupMenuNotes {
        id: popupMenuNotes

        onShowNotes: pdfPagesView.notesPaint = show
        onNoteActivate: pageStack.push(Qt.resolvedUrl("NotePage.qml"), {
                                           noteText: noteText,
                                           author: author,
                                           pageNumber: pageIndex + 1,
                                           color: color,
                                           noteId: annotationId,
                                           pdfPagesView: pdfPagesView,
                                           noteType: noteType,
                                           editable: pdfPagesView.isPageInPainted(pageIndex)
                                       })
        onOpened: pagesWithNotesModel.sourceModel = pdfPagesView.pagesWithNotesModel

        onCreateNote: {
            root.state = root.statesNames[4]
        }

        pagesWithNotesModel: PageWithNotesSortModel {
            filterParameter: PageWithNotesSortModel.Count
        }

        isShowNotes: pdfPagesView.notesPaint
        preferredWidth: applicationWindow.preferredWidthPopup
    }


    FileInfo {
        id: fileInfo

        objectName: "fileInfo"
        path: root.pdfPath
    }

    ShareAction {
        id: shareAction
        objectName: "shareAction"
    }

    states: [
        State {
            name: root.statesNames[0]

            PropertyChanges {
                target: pdfPagesView
                visible: true
            }

            PropertyChanges {
                target: appBar
                title: fileInfo.fileName
            }

            PropertyChanges {
                target: root
                noteCreatingMode: false
            }
        },
        State {
            name: root.statesNames[1]

            PropertyChanges {
                target: pdfPagesView
                visible: false
            }

            PropertyChanges {
                target: root
                noteCreatingMode: false
            }
        },
        State {
            name: root.statesNames[2]

            PropertyChanges {
                target: root
                pdfPath: ""
            }
            PropertyChanges {
                target: errorText
                text: qsTr("Could not open document")
            }
            PropertyChanges {
                target: pdfPagesView
                enabled: false
            }
            PropertyChanges {
                target: toolBar
                opened: false
            }
        },
        State {
            name: root.statesNames[3]

            PropertyChanges {
                target: root
                pdfPath: ""
            }
            PropertyChanges {
                target: errorText
                text: qsTr("File not found")
            }
            PropertyChanges {
                target: pdfPagesView
                enabled: false
            }
            PropertyChanges {
                target: toolBar
                opened: false
            }

            PropertyChanges {
                target: root
                noteCreatingMode: false
            }
        },
        State {
            name: root.statesNames[4]

            PropertyChanges {
                target: appBar
                title: qsTr("Hold a screen point to create a new node")
            }
            PropertyChanges {
                target: root
                noteCreatingMode: true
            }
            PropertyChanges {
                target: root
                isNotePointChosen: false
            }
            PropertyChanges {
                target: pdfPagesView
                visible: true
            }
        }
    ]
}
