// SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

import QtQuick 2.0
import Sailfish.Silica 1.0
import ru.auroraos.PdfViewer 1.0
import "../components"

Page {
    id: root

    property var pdfPagesView
    property string bakedSearchPhrase

    Component.onCompleted: bakedSearchPhrase = textField.text;

    signal searchResultSelected(int selectedResultIndex)
    signal searchEnded()

    allowedOrientations: Orientation.All

    PageHeader {
        id: header

        title: qsTr("Search by word")
    }

    TextField {
        id: textField

        anchors {
            top: header.bottom
            left: parent.left
            topMargin: Theme.horizontalPageMargin
        }

        text: pdfPagesView.searchPhraseModel.getPhrase()
        label: qsTr("Find a word")

        EnterKey.onClicked: {
            if (text) {
                textField.acceptInput();
            }
        }

        rightItem: IconButton {
            onClicked: {
                textField.acceptInput();
            }
            enabled: textField.text.trim().length > 0

            icon.source: "image://theme/icon-m-search"
        }

        function acceptInput() {
            bakedSearchPhrase = textField.text.trim();
            root.state = "searching";
            var onSearchPhraseModelChangedHandler = function (newModel) {
                pdfPagesView.onSearchPhraseModelChanged.disconnect(
                            onSearchPhraseModelChangedHandler);
                resultList.update();
                root.state = "readyForSearch";
                searchEnded();
            }
            pdfPagesView.onSearchPhraseModelChanged.connect(
                        onSearchPhraseModelChangedHandler);
            pdfPagesView.findPhrase(textField.text.trim());
            textField.focus = false;
            searchResultSelected(-1);
        }
    }

    BusyLabel {
        id: progressCircle
    }

    Button {
        id: cancelSearchButton

        onClicked: {
            root.state = "canceling"
            pdfPagesView.cancelSearch()
        }

        anchors {
            top: progressCircle.bottom
            topMargin: Theme.paddingMedium
            horizontalCenter: parent.horizontalCenter
        }

        visible: false
        text: qsTr("Cancel")
    }

    EmptyListStub {
        id: emptyConfigurationsStub

        anchors {
            top: textField.bottom
            left: parent.left
            right: parent.right
        }

        title: qsTr("Not found")
        visible: pdfPagesView.searchPhraseModel.rowCount() <= 0 && pdfPagesView.searchPhraseModel.getPhrase() !== ""
    }

    SilicaListView {
        id: resultList

        anchors {
            top: textField.bottom
            left: parent.left
            right: parent.right
            bottom: parent.bottom

            leftMargin: Theme.paddingLarge
            rightMargin: Theme.paddingLarge
            topMargin: Theme.itemSizeSmall * 2 / 3
            bottomMargin: Theme.itemSizeSmall / 2
        }
        model: pdfPagesView.searchPhraseModel
        spacing: Theme.paddingMedium
        clip: true

        delegate: ListItem {
            anchors.leftMargin: Theme.paddingLarge
            anchors.rightMargin: Theme.paddingLarge

            height: contextText.height
            contentHeight: height

            Text {
                id: contextText
                anchors {
                    left: parent.left
                    right: pageIndexText.left
                    rightMargin: Theme.paddingMedium
                }

                text: context
                wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                color: palette.primaryColor
                textFormat: Text.RichText
            }
            Text {
                id: pageIndexText

                anchors.right: parent.right
                text: phrasePageIndex + 1
                color: palette.primaryColor
                horizontalAlignment: Qt.AlignRight
            }
            onClicked: {
                searchResultSelected(index)
                pageStack.pop()
            }
        }

        function highlightText(txt) {
            var str = root.bakedSearchPhrase;
            var re = new RegExp("(" + str.replace(/[.*+?^${}()|[\]\\]/g, '\\$&') + ")", "gi");
            return txt.replace(re, "<span style='background-color: yellow; color: black;'>$1</span>");
        }
    }

    states: [
        State {
            name: "readyForSearch"

            PropertyChanges {
                target: resultList
                enabled: true
            }
            PropertyChanges {
                target: root
                backNavigation: true
            }
            PropertyChanges {
                target: textField
                enabled: true
            }
            PropertyChanges {
                target: progressCircle
                running: false
            }
            PropertyChanges {
                target: cancelSearchButton
                visible: false
            }
            PropertyChanges {
                target: resultList
                visible: true
            }
        },
        State {
            name: "searching"

            PropertyChanges {
                target: resultList
                visible: false
            }
            PropertyChanges {
                target: textField
                enabled: false
            }
            PropertyChanges {
                target: root
                backNavigation: false
            }
            PropertyChanges {
                target: progressCircle
                running: true
                text: qsTr("Searching for word") + " \"" + textField.text.trim() + "\""
            }
            PropertyChanges {
                target: cancelSearchButton
                visible: true
                enabled: true
            }
        },
        State {
            name: "canceling"

            PropertyChanges {
                target: resultList
                visible: false
            }
            PropertyChanges {
                target: textField
                enabled: false
            }
            PropertyChanges {
                target: root
                backNavigation: false
            }
            PropertyChanges {
                target: progressCircle
                running: true
                text: qsTr("Canceling search for word") + " \"" + textField.text.trim() + "\""
            }
            PropertyChanges {
                target: cancelSearchButton
                visible: true
                enabled: false
            }
        }
    ]
}
