// SPDX-FileCopyrightText: 2023 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

import QtQuick 2.0
import Sailfish.Silica 1.0

Page {
    id: root

    readonly property int penMinimumSize: 1
    readonly property int penMaximumSize: 60
    readonly property int colorCountPerRow: orientation & Orientation.LandscapeMask ? 12 : 4

    property string color: ""
    property int penSize: 1

    property bool showPenSizeSlider: true

    showNavigationIndicator: false
    allowedOrientations: Orientation.All

    signal chosen(int size, string color)

    Column {
        spacing: Theme.paddingLarge

        anchors {
            left: parent.left
            right: parent.right
            bottom: parent.bottom
            margins: Theme.paddingLarge
        }

        Rectangle {
            id: penSample

            height: root.penSize
            width: height
            radius: height / 2
            anchors.horizontalCenter: parent.horizontalCenter
            color: root.color
        }

        Item {
            id: penSize

            height: penSizeText.height + penSizeSlider.height + Theme.paddingMedium
            visible: showPenSizeSlider

            anchors {
                left: parent.left
                right: parent.right
            }

            Text {
                id: penSizeText

                text: qsTr("Pen size")
                color: Theme.highlightColor
                font.pixelSize: Theme.fontSizeLarge
                horizontalAlignment: Text.AlignHCenter

                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                }
            }

            Slider {
                id: penSizeSlider

                onValueChanged: {
                    root.penSize = value
                }

                minimumValue: penMinimumSize
                maximumValue: penMaximumSize
                value: root.penSize

                anchors {
                    top: penSizeText.bottom
                    topMargin: Theme.paddingMedium
                    left: parent.left
                    leftMargin: -2 * Theme.paddingLarge
                    right: parent.right
                    rightMargin: -2 * Theme.paddingLarge
                }
            }
        }

        Item {
            id: colorGrid

            height: gridText.height + gridView.height + Theme.paddingMedium

            anchors {
                left: parent.left
                right: parent.right
            }

            Text {
                id: gridText

                text: qsTr("Pen color")
                color: Theme.highlightColor
                font.pixelSize: Theme.fontSizeLarge
                horizontalAlignment: Text.AlignHCenter

                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                }
            }

            SilicaGridView {
                id: gridView

                height: contentHeight
                anchors {
                    top: gridText.bottom
                    topMargin: Theme.paddingMedium
                    left: parent.left
                    right: parent.right
                }

                cellWidth: width / colorCountPerRow
                cellHeight: cellWidth
                clip: true

                model: ["#e60003", "#e6007c", "#e700cc",
                    "#0077e7", "#01a9e7", "#00cce7",
                    "#00e696", "#00e600", "#99e600",
                    "#e3e601", "#e5bc00", "#e78601"]

                delegate: GridItem {
                    onClicked: root.color = modelData

                    Rectangle {
                        anchors.centerIn: parent
                        height: Math.min(Theme.itemSizeSmall, gridView.width / colorCountPerRow)
                        width: height
                        radius: height / 2
                        color: modelData
                        opacity: root.color === modelData ? 0.4 : 1
                    }

                    Rectangle {
                        visible: root.color === modelData
                        anchors.centerIn: parent
                        width: height
                        radius: height / 2
                        height: Math.min(Theme.itemSizeSmall, gridView.width / colorCountPerRow)
                        color: "transparent"
                        border {
                            color: modelData
                            width: 3
                        }
                    }

                    Image {
                        visible: root.color === modelData
                        anchors.centerIn: parent
                        source: "image://theme/icon-m-accept"
                    }
                }
            }
        }

        IconButton {
            id: closeButton

            onClicked: {
                root.chosen(root.penSize, root.color)
                pageStack.pop();
            }

            anchors.horizontalCenter: parent.horizontalCenter
            icon.source: "image://theme/icon-m-cancel?" + (pressed ? Theme.highlightColor : Theme.primaryColor)
        }
    }
}
