// SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

import QtQuick 2.0
import Sailfish.Silica 1.0

Dialog {
    id: root

    property color color
    property bool didSelectColor
    property color selectedColor

    allowedOrientations: Orientation.All

    onOpened: didSelectColor = false

    onDone: {
        if (result == DialogResult.Accepted && didSelectColor) {
            color = selectedColor
        }
    }

    function makeProcentFromReal(value) {
        return value * 100;
    }

    canAccept: didSelectColor

    SilicaFlickable {
        id: flick

        anchors.fill: parent
        contentHeight: mainColumn.height + Theme.paddingLarge

        VerticalScrollDecorator {}

        Column {
            id: mainColumn

            width: parent.width

            spacing: Theme.paddingMedium

            DialogHeader {
                id: header
                acceptText: qsTr("Choose color")
            }

            ColorPicker {
                id: picker

                colors: ["#e60003", "#e6007c", "#e700cc",
                    "#0077e7", "#01a9e7", "#00cce7",
                    "#00e696", "#00e600", "#99e600",
                    "#e3e601", "#e5bc00", "#e78601"]

                columns: Math.floor(width / Theme.itemSizeHuge)

                isPortrait: root.isPortrait
                onColorChanged: {
                    root.didSelectColor = true
                    const alpha = root.selectedColor.a;
                    root.selectedColor = picker.color
                    selectedColor.a = alpha;
                }
            }

            SectionHeader {
                text: qsTr("Transparent")
            }

            Row {
                width: parent.width

                anchors {
                    leftMargin: Theme.horizontalPageMargin
                    rightMargin: Theme.horizontalPageMargin
                }

                Slider {
                    id: alphaChannelSlider

                    width: parent.width - rectAlphaChannel.width - 2 * Theme.horizontalPageMargin

                    minimumValue: 0
                    maximumValue: 100

                    value: Math.round(selectedColor.a * 100)

                    valueText: Math.round(selectedColor.a * 100) + "%"

                    onValueChanged: {
                        root.selectedColor.a = value / 100;
                        root.didSelectColor = true;
                    }
                }

                Rectangle {
                    id: rectAlphaChannel

                    width: Theme.itemSizeSmall
                    height: Theme.itemSizeSmall

                    radius: width / 2

                    opacity: root.selectedColor.a

                    color: root.selectedColor
                }
            }
        }
    }
}
