// SPDX-FileCopyrightText: 2022-2024 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include <cmath>
#include "smoothline.h"

SmoothLine::SmoothLine(QList<QPointF> list, QObject *parent) : QObject(parent), m_list(list) { }

float SmoothLine::distance(const QPointF& pt1, const QPointF& pt2)
{
    float hd = (pt1.x() - pt2.x()) * (pt1.x() - pt2.x());
    float vd = (pt1.y() - pt2.y()) * (pt1.y() - pt2.y());
    return std::sqrt(hd + vd);
}

QPointF SmoothLine::getLineStart(const QPointF& pt1, const QPointF& pt2)
{
    QPointF pt;
    float rat = 3 / distance(pt1, pt2);
    if (rat > 0.5) {
        rat = 0.5;
    }
    pt.setX((1.0 - rat) * pt1.x() + rat * pt2.x());
    pt.setY((1.0 - rat) * pt1.y() + rat * pt2.y());
    return pt;
}

QPointF SmoothLine::getLineEnd(const QPointF& pt1, const QPointF& pt2)
{
    QPointF pt;
    float rat = 3 / distance(pt1, pt2);
    if (rat > 0.5) {
        rat = 0.5;
    }
    pt.setX(rat * pt1.x() + (1.0 - rat)*pt2.x());
    pt.setY(rat * pt1.y() + (1.0 - rat)*pt2.y());
    return pt;
}

QPainterPath SmoothLine::getPath(QList<QPointF> list, qreal xOff, qreal yOff, const float& factor)
{
    QList<QPointF> points;
    QPointF p;
    for (int i = 0; i < list.size(); i++) {
        p = QPointF(list.at(i).x() - xOff, list.at(i).y() - yOff);
        if (points.count() > 1 && (i < list.size() - 2) && (distance(points.last(), p) < factor)) {
            continue;
        }
        points.append(p);
    }

    QPainterPath path;

    QPointF pt1;
    QPointF pt2;

    for (int i = 0; i < points.count() - 1; i++) {
        pt1 = getLineStart(points[i], points[(i + 1) % points.size()]);
        if (i == 0) {
            path.moveTo(pt1);
        } else {
            path.quadTo(points[i], pt1);
        }
        pt2 = getLineEnd(points[i], points[(i + 1) % points.size()]);
        path.lineTo(pt2);
    }

    return path;
}
