// SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include "pdfwordmodel.h"

#include <QSharedPointer>
#include <baseannotation.h>
#include <pdfpageitem.h>

PdfWordModel::PdfWordModel(QObject *parent) : QAbstractListModel(parent) {  }

int PdfWordModel::rowCount(const QModelIndex &parent) const
{
    return parent.isValid() ? 0 : m_data.size();
}

QString PdfWordModel::getPhrase() {
    return m_phrase;
}

QVariant PdfWordModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid() || index.row() >= m_data.size())
        return {  };

    const auto *word = m_data.at(index.row());
    if (word == nullptr)
        return {  };
    switch (role) {
    case ValueRole : return QVariant::fromValue(word->value);
    case PageIndexRole: return QVariant::fromValue(word->pageIndex);
    case InRect: return QVariant::fromValue(word->rect);
    case ContextRole: return QVariant::fromValue(word->context);
    }

    return {  };
}

QHash<int, QByteArray> PdfWordModel::roleNames() const
{
    return {
        { ValueRole, "value" },
        { PageIndexRole, "phrasePageIndex" },
        { InRect, "rect" },
        { ContextRole, "context" }
    };
}


void PdfWordModel::clear()
{
    beginResetModel();
    qDeleteAll(m_data);
    m_data.clear();
    m_phrase = "";
    endResetModel();
}

int PdfWordModel::getPageIndex(int index)
{
    return m_data[index]->pageIndex;
}

int PdfWordModel::getSearchResultLocalIndex(int globalIndex)
{
    int localIndex = 0;
    const int currentPage = m_data[globalIndex]->pageIndex;
    for (int i = globalIndex - 1; i >= 0; i--) {
        int page = m_data[i]->pageIndex;
        if (currentPage != page)
            break;
        localIndex++;
    }
    return localIndex;
}

int PdfWordModel::getSearchResultCount()
{
    return m_data.size();
}

QString PdfWordModel::getSearchResutlText(int index)
{
    if (index >= m_data.size() || index < 0)
        return "";

    return m_data[index]->value;
}

void PdfWordModel::setNewData(QString phrase, const QVector<BaseWord *> &newData)
{
    clear();
    beginResetModel();
    m_data = newData;
    m_phrase = phrase;
    endResetModel();
}
