// SPDX-FileCopyrightText: 2022-2023 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include <QQuickWindow>
#include <QSGSimpleTextureNode>
#include <QPainter>

#include "baseannotation.h"

#include "pdfsimplenote.h"

#define TEXT_SEARCH_AUTHOR "TEXT_SEARCH"

PdfSimpleNote::PdfSimpleNote(QQuickItem *parent, QSharedPointer<BaseAnnotation> source) : QQuickItem(parent),
    m_highlighted(false),
    m_needUpdateImage(false),
    m_noteSource(source),
    m_isOverrideDefaultColor(false)
{
    setFlag(QQuickItem::ItemHasContents, true);
    setAcceptedMouseButtons(Qt::AllButtons);

    connect(this, &PdfSimpleNote::xChanged, this, &PdfSimpleNote::update);
    connect(this, &PdfSimpleNote::yChanged, this, &PdfSimpleNote::update);
    connect(this, &PdfSimpleNote::widthChanged, this, &PdfSimpleNote::update);
    connect(this, &PdfSimpleNote::heightChanged, this, &PdfSimpleNote::update);

    connect(parent, &QQuickItem::yChanged, this, &PdfSimpleNote::clearHighlight);
    connect(parent, &QQuickItem::xChanged, this, &PdfSimpleNote::clearHighlight);

    m_paper = _prepareTexture(HIGHLIGHT_OPACITY_NOTE_OFF);
    m_HighlightedPaper = _prepareTexture(HIGHLIGHT_OPACITY_NOTE_ON);
}

PdfSimpleNote::~PdfSimpleNote()
{
    m_noteSource.reset();
    this->deleteLater();
}

QSGNode *PdfSimpleNote::updatePaintNode(QSGNode *oldNode, QQuickItem::UpdatePaintNodeData *)
{
    if (height() <= 0 || width() <= 0)
        return nullptr;

    auto node = static_cast<QSGSimpleTextureNode *>(oldNode);

    if (node == nullptr) {
        node = new QSGSimpleTextureNode();
        node->setOwnsTexture(true);
    }

    if (m_needUpdateImage || node->texture() == nullptr) {
        if (node->texture() != nullptr)
            node->texture()->deleteLater();
        if (m_highlighted)
            node->setTexture(window()->createTextureFromImage(m_HighlightedPaper));
        else
            node->setTexture(window()->createTextureFromImage(m_paper));

        m_needUpdateImage = false;
    }

    node->setRect(boundingRect());

    return node;
}

QSharedPointer<BaseAnnotation> PdfSimpleNote::source() const
{
    return m_noteSource;
}

void PdfSimpleNote::setColor(QColor color)
{
    m_isOverrideDefaultColor = true;
    m_color = color;

    m_paper = _prepareTexture(HIGHLIGHT_OPACITY_NOTE_OFF);
    m_needUpdateImage = true;
    update();
}

void PdfSimpleNote::resetColor()
{
    m_isOverrideDefaultColor = false;

    m_paper = _prepareTexture(HIGHLIGHT_OPACITY_NOTE_OFF);
    m_needUpdateImage = true;
    update();
}

bool PdfSimpleNote::isSearchResult()
{
    return m_noteSource.data()->author == TEXT_SEARCH_AUTHOR;
}

QPointF PdfSimpleNote::getHighlightStartCoords()
{
    if (m_noteSource != nullptr && !m_noteSource->attachedPoints.isEmpty()) {
        qreal ratio = width() / m_noteSource->rect.width();
        qreal biasX =  m_noteSource->attachedPoints.first().x() - m_noteSource->rect.x();
        qreal biasY = m_noteSource->attachedPoints.first().y() - m_noteSource->rect.y();
        return QPointF(x() + biasX * ratio, y() + biasY * ratio);
    } else {
        return {};
    }
}

bool PdfSimpleNote::event(QEvent *event)
{
    if (opacity() < 1.0f)
        return QQuickItem::event(event);

    switch (event->type()) {
    case QEvent::MouseButtonPress:
        _setHighlight(true);
        return true;

    case QEvent::MouseButtonRelease:
        if (m_noteSource != nullptr)
            emit triggered(m_noteSource->content, m_noteSource->author, m_noteSource->annotationId, m_noteSource->color, m_noteSource->getTypeAsString());
        _setHighlight(false);
        break;

    case QEvent::MouseMove:
        _setHighlight(false);
        break;

    default:
        break;
    }

    return QQuickItem::event(event);
}

void PdfSimpleNote::_setHighlight(bool highlight)
{
    if (!m_noteSource)
        return;

    if (m_highlighted != highlight) {
        m_highlighted = highlight;
        m_needUpdateImage = true;
        update();
    }
}

QImage PdfSimpleNote::_prepareTexture(int alpha)
{
    if (m_noteSource->type == BaseAnnotation::AnnotationType::HighLight) {
        QImage paper = QImage(m_noteSource->rect.width(), m_noteSource->rect.height(),
                              QImage::Format_RGBA8888);
        paper.fill(Qt::transparent);
        QPixmap px = QPixmap::fromImage(paper);
        QPainter painter(&px);
        painter.setRenderHint(QPainter::HighQualityAntialiasing);
        painter.setPen(Qt::NoPen);

        QColor color;
        if (m_isOverrideDefaultColor) {
            color = m_color;
        } else if (m_noteSource->color.isValid()) {
            color = m_noteSource->color;
        } else {
            color = qRgba(21, 222, 225, 0);
        }
        color.setAlpha(alpha);

        painter.setBrush(color);

        for (const auto attachedRect : m_noteSource->attachedPoints) {
            QRectF rect = attachedRect;
            rect.moveTo(attachedRect.x() - m_noteSource->rect.x(), attachedRect.y() - m_noteSource->rect.y());
            painter.drawRect(rect);

        }
        painter.end();
        return px.transformed(QTransform().scale(1, -1)).toImage();
    } else {
        QImage paper = QImage(1, 1, QImage::Format_RGBA8888);
        auto fillColor = m_noteSource->color.isValid()
                ? m_noteSource->color
                : qRgba(21, 222, 225, 0);
        fillColor.setAlpha(alpha);
        paper.fill(fillColor);
        return paper;
    }
}

void PdfSimpleNote::clearHighlight()
{
   _setHighlight(false);
}
