// SPDX-FileCopyrightText: 2022 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause
#include <QQuickWindow>
#include <QSGFlatColorMaterial>
#include <QSGGeometry>
#include <QSGSimpleTextureNode>

#include <QPainter>
#include "baseannotation.h"

#include "smoothline.h"
#include "pdfsimpleannotation.h"

PdfSimpleAnnotation::PdfSimpleAnnotation(QQuickItem *parent, QSharedPointer<BaseAnnotation> source) : QQuickItem(parent),
    m_annotationSource(source),
    m_highlighted(false),
    m_needUpdateImage(false)
{
    setFlag(QQuickItem::ItemHasContents, true);
    setAcceptedMouseButtons(Qt::AllButtons);

    connect(this, &PdfSimpleAnnotation::xChanged, this, &PdfSimpleAnnotation::update);
    connect(this, &PdfSimpleAnnotation::yChanged, this, &PdfSimpleAnnotation::update);
    connect(this, &PdfSimpleAnnotation::widthChanged, this, &PdfSimpleAnnotation::update);
    connect(this, &PdfSimpleAnnotation::heightChanged, this, &PdfSimpleAnnotation::update);

    connect(parent, &QQuickItem::yChanged, this, &PdfSimpleAnnotation::clearHighlight);
    connect(parent, &QQuickItem::xChanged, this, &PdfSimpleAnnotation::clearHighlight);

    if (m_annotationSource->type == BaseAnnotation::AnnotationType::Ink
        || m_annotationSource->type == BaseAnnotation::AnnotationType::Polygon
        || m_annotationSource->type == BaseAnnotation::AnnotationType::Polyline) {
        m_needUpdateImage = true;
        drawInk();
    } else {
        // отрисовка обычных простых аннотаций в виде квадратов.
        m_paper = QImage(1, 1, QImage::Format_RGBA8888);
        m_paper.fill(qRgba(1, 222, 121, HIGHLIGHT_OPACITY_ANNOT_OFF));
    }
}

PdfSimpleAnnotation::~PdfSimpleAnnotation()
{
    m_annotationSource.reset();
    this->deleteLater();
}

void PdfSimpleAnnotation::drawInk()
{
    QImage paper = QImage(m_annotationSource->rect.width(), m_annotationSource->rect.height(), QImage::Format_RGBA8888);
    paper.fill(qRgba(255, 255, 255, 0));
    qreal x = m_annotationSource->rect.x();
    qreal y = m_annotationSource->rect.y();

    QPixmap px = QPixmap::fromImage(paper);
    QPainter p(&px);
    QPen pen(m_annotationSource->color.isValid() ? m_annotationSource->color : Qt::red,
                      m_annotationSource->lineWidth > 0 ? m_annotationSource->lineWidth : 2);
    pen.setCapStyle(Qt::PenCapStyle::RoundCap);
    p.setPen(pen);
    p.setRenderHint(QPainter::HighQualityAntialiasing, true);
    p.setRenderHint(QPainter::SmoothPixmapTransform, true);
    QPainterPath path;

    for (int i = 0; i < m_annotationSource->pointLists.size(); i++) {
        SmoothLine line(m_annotationSource->pointLists.at(i));
        path.addPath(line.getPath(m_annotationSource->pointLists.at(i), x, y));
    }

    p.drawPath(path);
    p.end();
    m_paper = px.transformed(QTransform().scale(1, -1)).toImage();
}

QSGNode *PdfSimpleAnnotation::updatePaintNode(QSGNode *oldNode, QQuickItem::UpdatePaintNodeData *)
{
        if (height() <= 0 || width() <= 0)
            return nullptr;

        auto node = static_cast<QSGSimpleTextureNode *>(oldNode);

        if (node == nullptr) {
            node = new QSGSimpleTextureNode();
            node->setOwnsTexture(true);
        }

        if (m_needUpdateImage || node->texture() == nullptr) {
            if (node->texture() != nullptr)
                node->texture()->deleteLater();

            node->setTexture(window()->createTextureFromImage(m_paper));

            m_needUpdateImage = false;
        }

        node->setRect(boundingRect());

        return node;
}

bool PdfSimpleAnnotation::event(QEvent *event)
{
    if (opacity() < 1.0f)
        return QQuickItem::event(event);

    switch (event->type()) {
    case QEvent::MouseButtonPress:
        _setHighlight(true);
        return true;

    case QEvent::MouseButtonRelease:
        if (m_annotationSource != nullptr)
            emit triggered(m_annotationSource);
        _setHighlight(false);
        break;

    case QEvent::MouseMove:
        _setHighlight(false);
        break;

    default:
        break;
    }

    return QQuickItem::event(event);
}

void PdfSimpleAnnotation::_setHighlight(bool highlight)
{
    if (m_highlighted != highlight) {
        m_highlighted = highlight;
        m_needUpdateImage = true;
        if (m_annotationSource->type == BaseAnnotation::AnnotationType::Ink){
          //  drawInk();
        } else {
            m_paper.fill(qRgba(1, 222, 121, highlight ? HIGHLIGHT_OPACITY_ANNOT_ON : HIGHLIGHT_OPACITY_ANNOT_OFF));
        }
        update();
    }
}

void PdfSimpleAnnotation::clearHighlight()
{
    _setHighlight(false);
}

QSharedPointer<BaseAnnotation> PdfSimpleAnnotation::source() const
{
    return m_annotationSource;
}
