// SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#ifndef PDFDOCUMENTITEM_H
#define PDFDOCUMENTITEM_H

#include <QObject>
#include <QSharedPointer>
#include <QHash>
#include <QSet>
#include <QVector>
#include <QPointer>
#include <amberpdf/pdfpagetexts.h>
#include <amberpdf/pdfword.h>
#include <baseword.h>

#include "basedocument.h"
#include "pdfpageitem.h"
#include "pagepreloader.h"

class PdfPageTexts;
class PdfPage;
class PdfDocument;
class PdfDocumentItem : public BasePdfDocument
{
    Q_OBJECT

public:
    explicit PdfDocumentItem(QObject *parent = nullptr);
    ~PdfDocumentItem();

    QString path() const override;
    QSizeF pageSize(int pageNumber) const override;
    Q_INVOKABLE int count() const override;
    void loadAllPages() override;
    QSharedPointer<BasePage> loadPage(int pageIndex) override;
    void startLoadBookmarks() const override;
    QVector<BaseBookmark *> bookmarks() const override;
    int fileVersion() const override;
    bool saveDocumentAs(const QString &path) const override;    
    void findPhrase(QString phrase) override;
    QVector<BaseWord *> foundPhrases() override;
    void startLoadUserBookmarks() const override;
    QString findStartingTextForBookmark(int pageIndex) override;
    QVector<UserBookmark *> userBookmarks() override;
    void addUserBookmark(int pageNumber, const QString &pageText) override;
    void deleteUserBookmark(int pageNumber) override;
    void generateFileName() override;
    void refreshBookmarksFile(const QString &filePath) override;
    QSharedPointer<PdfDocument> document() override;
    PdfPageTexts* pageTextsModel();

public slots:
    void setPath(const QString &path) override;

private slots:
    void updateTextSearchResult(const QVector<BaseWord *>& phrases, const QString phraseToFind);
    void removeCanceledSearchResult(const QString phraseToFind);
    void updateFileId(const QString &newFileName);
    void updateFileHash(const QString &newFileHash);

private:
    void onPagePreloaderDone(int loadedPageIndex, PageLoadStatus loadStatus);
    void generateFileHash();

private:
    QSharedPointer<PdfDocument> m_pdfiumDocument;
    QHash<int, QSizeF> m_pageSizes;
    QHash<int, QSharedPointer<BasePage>> m_loadedPages;
    QSet<int> m_pagesInProcess;
    QVector<BaseBookmark *> m_baseBookmarks;
    QVector<UserBookmark *> m_userBookmarks;
    QString m_fileID;
    QString m_fileHash;
    QString m_fileName;
    QString m_savedFileName;
    bool m_refreshingBookmarksFile = false;
    QVector<BaseWord *> m_foundPhrases;
    QPointer<PagePreloader> m_preloaderAllPage;
    PdfPageTexts* m_pageTextsModel;
};

#endif // PDFDOCUMENTITEM_H
