// SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include <QFile>
#include <QFutureWatcher>
#include <QSizeF>
#include <QThreadPool>
#include <QVector>
#include <QPair>
#include <QFileInfo>
#include <QMap>

#include <amberpdf/pdfannotation.h>
#include <amberpdf/pdfdocument.h>
#include <amberpdf/pdfbookmark.h>
#include <amberpdf/userpdfbookmark.h>
#include <pdfdocumenthashworker.h>

#include "baseannotation.h"
#include "pdfsearchresultsremover.h"
#include "pagessizesloader.h"
#include "pagepreloader.h"
#include "basepage.h"
#include "basebookmark.h"
#include "userbookmark.h"

#include "pdfdocumentitem.h"

PdfDocumentItem::PdfDocumentItem(QObject *parent)
    : BasePdfDocument(parent)
{
    qRegisterMetaType<QHash<int, QSizeF>>();

    m_status = DocumentStatus::Null;
    m_pageTextsModel = new PdfPageTexts(this);
}

PdfDocumentItem::~PdfDocumentItem()
{
    if (m_preloaderAllPage)
        m_preloaderAllPage->cancel();

    m_pdfiumDocument.clear();
    m_pageSizes.clear();
    m_loadedPages.clear();
    m_pagesInProcess.clear();
    m_baseBookmarks.clear();
    m_foundPhrases.clear();
}

QString PdfDocumentItem::path() const
{
    return m_pdfiumDocument ? m_pdfiumDocument->path() : QStringLiteral("");
}

QSizeF PdfDocumentItem::pageSize(int pageNumber) const
{
    if (m_status != DocumentStatus::Ready)
        return {  };

    return m_pageSizes.value(pageNumber);
}

int PdfDocumentItem::count() const
{
    return m_pdfiumDocument ? m_pdfiumDocument->pageCount() : -1;
}

void PdfDocumentItem::loadAllPages()
{
    if (m_preloaderAllPage)
        return;

    QVector<int> notLoadedPage;

    for (int i = 0; i < count(); i++) {
        if (!m_loadedPages.contains(i) && !m_pagesInProcess.contains(i))
            notLoadedPage.append(i);
    }

    if (notLoadedPage.empty()) {
        return;
    }

    QVector<QPair<int, int>> rangesToLoad;

    int startPage = notLoadedPage[0];
    int endPage = startPage;

    for (int i = 1; i < notLoadedPage.size(); ++i) {
        if (notLoadedPage[i] == endPage + 1) {
            endPage = notLoadedPage[i];
        } else {
            rangesToLoad.push_back(QPair(startPage, endPage));
            startPage = endPage = notLoadedPage[i];
        }
    }

    rangesToLoad.push_back(QPair(startPage, endPage));

    for (const auto &range : rangesToLoad) {
        m_preloaderAllPage = new PagePreloader(m_pdfiumDocument, range.first, range.second - range.first);
        connect(m_preloaderAllPage, &PagePreloader::done, this, [&] (int loadedPageIndex, PageLoadStatus loadStatus) {
            if (loadStatus == PageLoadStatus::Success) {
                if (!m_loadedPages.contains(loadedPageIndex))
                    m_loadedPages.insert(loadedPageIndex, QSharedPointer<BasePage>(
                                             new PdfPageItem(m_pdfiumDocument->page(loadedPageIndex))));
            }

            emit pageLoaded(loadedPageIndex, loadStatus);
        });
        QThreadPool::globalInstance()->start(m_preloaderAllPage, -1);
        for (int i = range.first; i < range.second; i++)
            m_pagesInProcess.insert(i);
    }
}

QSharedPointer<BasePage> PdfDocumentItem::loadPage(int pageIndex)
{
    if (pageIndex < 0 || pageIndex >= count())
        return {  };

    if (m_loadedPages.contains(pageIndex)){
        return m_loadedPages.value(pageIndex);
    }

    if (m_pagesInProcess.contains(pageIndex)){
        return {  };
    }
    m_pagesInProcess.insert(pageIndex);

    auto *pageLoadingWatcher = new QFutureWatcher<QSharedPointer<PdfPage>>();
    connect(pageLoadingWatcher,  &QFutureWatcher<QSharedPointer<PdfPage>>::finished,
            this,  [pageLoadingWatcher, pageIndex, this] () {
        m_loadedPages.insert(pageIndex, QSharedPointer<BasePage>(
                                 new PdfPageItem(m_pdfiumDocument->page(pageIndex))));
        pageTextsModel()->addPage(pageIndex, m_loadedPages.value(pageIndex)->text());
        emit pageLoaded(pageIndex, PageLoadStatus::Success);
        pageLoadingWatcher->deleteLater();
    });
    pageLoadingWatcher->setFuture(m_pdfiumDocument->page(pageIndex));

    return {  };
}

void PdfDocumentItem::startLoadBookmarks() const
{
    auto *bookmarksWatcher = new QFutureWatcher<QVector<PdfBookmark>>();
    connect(bookmarksWatcher,  &QFutureWatcher<QVector<PdfBookmark>>::finished,
            this,  [bookmarksWatcher, ctx = const_cast<PdfDocumentItem *>(this)]() {
        if (bookmarksWatcher == nullptr)
            return;

        if (bookmarksWatcher->isFinished() && !bookmarksWatcher->isCanceled()) {
            auto bookmarks = bookmarksWatcher->result();
            if (bookmarks.isEmpty()) {
                bookmarksWatcher->deleteLater();
                return;
            }

            qDeleteAll(ctx->m_baseBookmarks);
            ctx->m_baseBookmarks.clear();

            for (const auto &bookmark : bookmarks) {
                ctx->m_baseBookmarks.push_back(new BaseBookmark{ bookmark.title, bookmark.page,
                                                                 bookmark.level, bookmark.locationInPage
                                               });
            }

            emit ctx->bookmarksLoaded();
        }

        bookmarksWatcher->deleteLater();
    });
    bookmarksWatcher->setFuture(m_pdfiumDocument->bookmarks());
}

QVector<BaseBookmark *> PdfDocumentItem::bookmarks() const
{
    return m_baseBookmarks;
}

int PdfDocumentItem::fileVersion() const
{
    return (m_pdfiumDocument.isNull() ? -1 : m_pdfiumDocument->fileVersion());
}

QVector<BaseWord *>  PdfDocumentItem::foundPhrases() {
    return m_foundPhrases;
}

void PdfDocumentItem::findPhrase(QString phrase)
{
    auto *searchWatcher = new QFutureWatcher<PdfPageTexts*>();
    connect(this, &PdfDocumentItem::searchCanceled, searchWatcher, &QFutureWatcher<QVector<QSharedPointer<PdfWord>>>::cancel);
    connect(searchWatcher, &QFutureWatcher<QVector<QSharedPointer<PdfWord>>>::finished,
            this,  [searchWatcher, phrase, ctx = const_cast<PdfDocumentItem *>(this)]() {
        if (searchWatcher == nullptr)
            return;

        QVector<BaseWord*> phrases;
        if (searchWatcher->isFinished() && !searchWatcher->isCanceled()) {
            auto foundPhrases = searchWatcher->result()->phrasesForModel();

            if (foundPhrases.isEmpty()) {
                ctx->updateTextSearchResult(phrases, phrase);
                searchWatcher->deleteLater();
                return;
            }

            for (const auto &phrase : foundPhrases) {
                phrases.append(new BaseWord{phrase->value(), phrase->index(), phrase->rect(), phrase->attachedRects(), phrase->context()});
            }

            searchWatcher->deleteLater();
        }
        ctx->updateTextSearchResult(phrases, phrase);
    });
    connect(searchWatcher, &QFutureWatcher<QVector<QSharedPointer<PdfWord>>>::canceled,
            this,  [searchWatcher, phrase, ctx = const_cast<PdfDocumentItem *>(this)]() {
        ctx->removeCanceledSearchResult(phrase);
        searchWatcher->deleteLater();
    });
    searchWatcher->setFuture(m_pdfiumDocument->searchPhrase(phrase, m_pageTextsModel));
}

void PdfDocumentItem::updateTextSearchResult(const QVector<BaseWord *>& phrases, const QString phraseToFind)
{
    m_foundPhrases.clear();
    m_foundPhrases.append(phrases);
    emit BasePdfDocument::phraseFound(m_foundPhrases, phraseToFind);
}

void PdfDocumentItem::removeCanceledSearchResult(const QString phraseToFind)
{
    if (!m_foundPhrases.empty())
        m_foundPhrases.clear();

    auto pdfSearchResultsRemover = new PdfSearchResultsRemover(m_pdfiumDocument);
    connect(pdfSearchResultsRemover, &PdfSearchResultsRemover::done, this, [this, phraseToFind](){
        emit BasePdfDocument::phraseFound(m_foundPhrases, phraseToFind);
    });
    QThreadPool::globalInstance()->start(pdfSearchResultsRemover);
}

/*!
 * \brief Gets first n words from certain page.
 * \param pageIndex Page index.
 * \return First n words.
 */
QString PdfDocumentItem::findStartingTextForBookmark(int pageIndex)
{
    QVector<BaseWord*> phrasesForModel;
    auto page = m_pdfiumDocument->page(pageIndex);
    page.waitForFinished();
    auto wordsF = page.result()->words();
    wordsF.waitForFinished();
    auto words = wordsF.result();

    QList<QString> totalList;
    for (int i = 0; i < std::min(5, words.size()); ++i) {
        totalList.append(static_cast<PdfWord*>(words[i])->value());
    }

    return totalList.join(" ");
}

/*!
 * \brief Starts the process of loading user bookmarks from the user bookmarks
 * json file.
 */
void PdfDocumentItem::startLoadUserBookmarks() const
{
    auto *bookmarksWatcher = new QFutureWatcher<QVector<UserPdfBookmark>>();
    connect(bookmarksWatcher,  &QFutureWatcher<QVector<UserPdfBookmark>>::finished,
            this,  [bookmarksWatcher, ctx = const_cast<PdfDocumentItem *>(this)]() {
        if (bookmarksWatcher == nullptr)
            return;

        if (bookmarksWatcher->isFinished() && !bookmarksWatcher->isCanceled()) {
            auto bookmarks = bookmarksWatcher->result();

            qDeleteAll(ctx->m_userBookmarks);
            ctx->m_userBookmarks.clear();

            for (const auto &bookmark : bookmarks) {
                ctx->m_userBookmarks.push_back(new UserBookmark{bookmark.page, bookmark.text});
            }

            emit ctx->userBookmarksLoaded();
        }
        bookmarksWatcher->deleteLater();
    });
    bookmarksWatcher->setFuture(m_pdfiumDocument->userBookmarks(m_fileName));
}

/*!
 * \brief Returns vector with user bookmarks.
 * \return QVector with user bookmarks.
 */
QVector<UserBookmark *> PdfDocumentItem::userBookmarks()
{
    return m_userBookmarks;
}

/*!
 * \brief Adds new user bookmark for selected page.
 * \param pageNumber Page number.
 * \param pageText Page text.
 */
void PdfDocumentItem::addUserBookmark(int pageNumber, const QString &pageText)
{
    auto *watcher = new QFutureWatcher<bool>();
    connect(watcher, &QFutureWatcher<bool>::finished, this, [this, watcher]() {
        if (watcher == nullptr)
            return;

        if (watcher->isFinished() && !watcher->isCanceled())
        {
            startLoadUserBookmarks();
        }

        watcher->deleteLater();
    });
    auto future = m_pdfiumDocument->addUserBookmark(pageNumber, pageText, m_fileName);
    watcher->setFuture(future);
}

/*!
 * \brief Deletes user bookmark from selected page.
 * \param pageNumber Page number.
 */
void PdfDocumentItem::deleteUserBookmark(int pageNumber)
{
    auto *watcher = new QFutureWatcher<bool>();
    connect(watcher, &QFutureWatcher<bool>::finished, this, [this, watcher]() {
        if (watcher == nullptr)
            return;

        if (watcher->isFinished() && !watcher->isCanceled())
        {
            startLoadUserBookmarks();
        }

        watcher->deleteLater();
    });
    watcher->setFuture(m_pdfiumDocument->deleteUserBookmark(pageNumber, m_fileName));
}

/*!
 * \brief This method generates user bookmarks file name based on document id or
 * hash. Further, if the document has id than no hash sum will be counted and the
 * user bookmarks json file will be named by document id. Otherwise, the file
 * will be named by document hash.
 */
void PdfDocumentItem::generateFileName()
{
    auto *watcher = new QFutureWatcher<QString>();
    connect(watcher, &QFutureWatcher<QString>::finished, this, [this, watcher]() {
        if (watcher == nullptr) {
            return;
        }
        if (watcher->isFinished() && !watcher->isCanceled())
        {
            auto id = watcher->result();
            if (id.isEmpty()) {
                generateFileHash();
            } else {
                updateFileId(id);
            }
        }
        watcher->deleteLater();
    });
    watcher->setFuture(m_pdfiumDocument->getFileId());
}

/*!
 * \brief Enables file hash worker to find current file hash.
 */
void PdfDocumentItem::generateFileHash()
{
    auto fileHashWorker = new PdfDocumentHashWorker(path());
    connect(fileHashWorker, &PdfDocumentHashWorker::fileHashCounted, this, &PdfDocumentItem::updateFileHash);
    QThreadPool::globalInstance()->start(fileHashWorker);
}

/*!
 * \brief Refreshes known to active document object user bookmarks file name
 * based on newest document id.
 * \param newFileName New document id.
 */
void PdfDocumentItem::updateFileId(const QString &newFileName)
{
    m_fileName = newFileName;
    m_fileID = newFileName;

    if (!m_refreshingBookmarksFile) {
        // If the refreshing user bookmarks file name process is not active.
        startLoadUserBookmarks();
    } else {
        // If the refreshing user bookmarks file name process is active.
        auto *watcher = new QFutureWatcher<bool>();
        connect(watcher, &QFutureWatcher<bool>::finished, this, [this, watcher]() {
            if (watcher == nullptr) {
                return;
            }
            if (watcher->isFinished() && !watcher->isCanceled()) {
                m_refreshingBookmarksFile = false;
                m_savedFileName = "";
                startLoadUserBookmarks();
            }
            watcher->deleteLater();
        });
        watcher->setFuture(m_pdfiumDocument->renameBookmarksFile(m_savedFileName, m_fileID));
    }
}

/*!
 * \brief Refreshes known to current document object user bookmarks file name
 * based on newest document hash.
 * \param newFileName New document hash.
 */
void PdfDocumentItem::updateFileHash(const QString &newFileHash)
{
    m_fileName = newFileHash;
    m_fileHash = newFileHash;

    if (!m_refreshingBookmarksFile) {
        startLoadUserBookmarks();
    } else {
        // If the refreshing user bookmarks file name process is active.
        auto *watcher = new QFutureWatcher<bool>();
        connect(watcher, &QFutureWatcher<bool>::finished, this, [this, watcher]() {
            if (watcher == nullptr) {
                return;
            }
            if (watcher->isFinished() && !watcher->isCanceled()) {
                m_refreshingBookmarksFile = false;
                m_savedFileName = "";
                startLoadUserBookmarks();
            }
            watcher->deleteLater();
        });
        watcher->setFuture(m_pdfiumDocument->renameBookmarksFile(m_savedFileName, m_fileHash));
    }
}

/*!
 * \brief Refreshes user bookmarks file name if the original document had no id.
 * \param filePath Path to the document copy.
 */
void PdfDocumentItem::refreshBookmarksFile(const QString &filePath)
{
    m_savedFileName = m_fileName;
    auto *watcher = new QFutureWatcher<QString>();
    connect(watcher, &QFutureWatcher<QString>::finished, this, [this, watcher]() {
        if (watcher == nullptr) {
            return;
        }
        if (watcher->isFinished() && !watcher->isCanceled()) {
            auto fileID = watcher->result();
            if (!fileID.isEmpty()) {
                // If the document copy file has id than update user bookmarks file name.
                m_refreshingBookmarksFile = true;
                updateFileId(fileID);
            } else {
                // If the document copy has no id than update user bookmarks file name
                // using file hash.
                m_refreshingBookmarksFile = true;
                generateFileHash();
            }
        }
        watcher->deleteLater();
    });
    watcher->setFuture(m_pdfiumDocument->getNewFileId(filePath));
}

QSharedPointer<PdfDocument> PdfDocumentItem::document()
{
    return m_pdfiumDocument;
}

PdfPageTexts *PdfDocumentItem::pageTextsModel()
{
    return m_pageTextsModel;
}

bool PdfDocumentItem::saveDocumentAs(const QString &path) const
{
    if (!m_pdfiumDocument)
        return false;

    if (!QFile::exists(path))
        return m_pdfiumDocument->saveDocumentAs(path);

    static QString firstSuffix = QStringLiteral(".amber_bak1");
    auto backupPathFirst = path + firstSuffix;

    auto saveResult =  m_pdfiumDocument->saveDocumentAs(backupPathFirst);
    if (!saveResult)
        return false;

    static QString seconsSuffix = QStringLiteral(".amber_bak2");
    auto backupPathSecond = path + seconsSuffix;
    saveResult = QFile::rename(path, backupPathSecond);
    if (!saveResult)
        return false;

    saveResult = QFile::rename(backupPathFirst, path);
    if (!saveResult)
        return false;

    QFile::remove(backupPathFirst);
    QFile::remove(backupPathSecond);

    return true;
}

void PdfDocumentItem::setPath(const QString &path)
{
    if (m_pdfiumDocument && (m_pdfiumDocument->path() == path))
        return;

    if (m_status != DocumentStatus::Loading) {
        m_status = DocumentStatus::Loading;
        emit statusChanged(m_status);
    }

    m_pdfiumDocument.reset(new PdfDocument());
    m_pageSizes.clear();
    m_loadedPages.clear();
    m_pagesInProcess.clear();
    m_baseBookmarks.clear();
    m_foundPhrases.clear();

    if (m_pdfiumDocument == nullptr) {
        m_status = DocumentStatus::Error;
        emit statusChanged(m_status);
        return;
    }

    connect(m_pdfiumDocument.data(), &PdfDocument::statusChanged, this, [&](PdfDocument::DocumentStatus status) {
        if (status == PdfDocument::Success || status == PdfDocument::Loading) {
            m_pdfiumDocument->pageCount();
            return;
        }

        m_status = DocumentStatus::Error;
        emit statusChanged(m_status);
    });

    connect(m_pdfiumDocument.data(), &PdfDocument::pageCountChanged, this, [&](int pagesCount) {
        if (m_pdfiumDocument->status() != PdfDocument::Success)
            return;

        if (pagesCount < 0) {
            m_status = DocumentStatus::Error;
            emit statusChanged(m_status);
            return;
        }

        auto pageLoader = new PagesSizesLoader(m_pdfiumDocument);
        connect(pageLoader, &PagesSizesLoader::done, this, [this](QHash<int, QSizeF> sizes) {
            m_pageSizes.swap(sizes);
            m_status = DocumentStatus::Ready;
            emit statusChanged(m_status);
        });
        QThreadPool::globalInstance()->start(pageLoader);
    });

    m_pdfiumDocument->loadDocument(path);

    emit pathChanged(path);
}

void PdfDocumentItem::onPagePreloaderDone(int loadedPageIndex, PageLoadStatus loadStatus)
{
    if (m_pagesInProcess.isEmpty()) {
        // the document was probably changed recently
        // drop loading the pages of the previous document
        emit pageLoaded(loadedPageIndex, PageLoadStatus::Fail);
        return;
    }
    if (loadStatus == PageLoadStatus::Success) {
        auto pdfPageFuture = m_pdfiumDocument->page(loadedPageIndex);
        if (!pdfPageFuture.resultCount()) return;
        auto pdfPageItem = new PdfPageItem(pdfPageFuture);
        auto basePage =  QSharedPointer<BasePage>(pdfPageItem);
        m_loadedPages.insert(loadedPageIndex, basePage);
        m_pagesInProcess.remove(loadedPageIndex);
    }

    emit pageLoaded(loadedPageIndex, loadStatus);
}
