/****************************************************************************
**
** Copyright (C) 2021 - 2023 Open Mobile Platform LLC.
** Contact: https://community.omprussia.ru/open-source
**
** This file is part of the AmberPDF project.
**
** $QT_BEGIN_LICENSE:BSD$
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Open Mobile Platform LLC copyright holder nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <pdfium/fpdfview.h>

#include "pdftaskqueue.h"
#include "pdfdocumentholder.h"
#include "pdfpagesizetask.h"

/*!
 * \class PdfPageSizeTask
 * \brief The PdfPageSizeTask class is a task for recieving size of the page.
 * \inmodule AmberPDF
 * \ingroup tasks
 */

/*!
 * Parametrized constructor.
 * \a pageNumber is index of page in document.
 * \a documentHolder is QSharedPointer to PdfDocumentHolder object with document.
 * \a interface is QFutureInterface for future tasks.
 */
PdfPageSizeTask::PdfPageSizeTask(int pageNumber,
                                 const QSharedPointer<PdfDocumentHolder> &documentHolder,
                                 const QFutureInterface<QSizeF> &interface) :
    m_pageNumber(pageNumber),
    m_interface(interface),
    m_documentHolder(documentHolder)
{  }

PdfPageSizeTask::~PdfPageSizeTask() = default;

/*!
 * Starts the process of recieving size of the page.
 */
void PdfPageSizeTask::run()
{
    if (m_interface.isCanceled() || PdfTaskQueue::instance().blockedId().contains(m_documentHolder->id()))
        return;

    FS_SIZEF pageSize;
    FPDF_GetPageSizeByIndexF(m_documentHolder->document().data(), m_pageNumber, &pageSize);
    m_size = QSizeF{pageSize.width, pageSize.height};

    m_interface.reportFinished(&m_size);
}

/*!
 * Returns document id.
 */
int PdfPageSizeTask::id() const
{
    return m_documentHolder->id();
}

/*!
 * Cancels task.
 */
void PdfPageSizeTask::cancel() {  }
