// SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include <QStandardPaths>
#include <QFile>
#include <QJsonDocument>
#include <QJsonObject>

#include "pdfdocumentholder.h"
#include "pdftaskqueue.h"

#include "pdfdocumentuserbookmarksloadtask.h"

/*!
 * Parses json file to extract user bookmarks.
 * \a bookmarks is user bookmarks QVector.
 * \a fileName is file name.
 */
static void parseUserBookmarks(QVector<UserPdfBookmark> *bookmarks, const QString &fileName) {
    QString cachePath = QStandardPaths::writableLocation(QStandardPaths::CacheLocation);
    QString filePath = cachePath + "/" + fileName;

    QFile file(filePath);
    if (!file.open(QIODevice::ReadOnly)) {
        return;
    }

    QByteArray jsonData = file.readAll();
    file.close();

    QJsonDocument doc = QJsonDocument::fromJson(jsonData);
    if (!doc.isObject()) {
        return;
    }

    QJsonObject rootObj = doc.object();
    if (!rootObj.contains("bookmarks") || !rootObj["bookmarks"].isObject()) {
        return;
    }

    QJsonObject bookmarksObj = rootObj["bookmarks"].toObject();
    for (auto item = bookmarksObj.begin(); item != bookmarksObj.end(); ++item) {
        int page = item.key().toInt();
        UserPdfBookmark bookmark;
        bookmark.page = page;
        bookmark.text = item.value().toString();
        bookmarks->push_back(bookmark);
    }
    std::sort(bookmarks->begin(), bookmarks->end());
}

bool operator<(const UserPdfBookmark& a, const UserPdfBookmark& b) { return a.page < b.page; }

/*!
 * Parametrized constructor.
 * \a documentHolder is QSharedPointer to PdfDocumentHolder object with document.
 * \a interface is QFutureInterface for future tasks.
 * \a fileName is name of file.
 */
PdfDocumentUserBookmarksLoadTask::PdfDocumentUserBookmarksLoadTask(
        const QSharedPointer<PdfDocumentHolder> &documentHolder,
        const QFutureInterface<QVector<UserPdfBookmark> > &interface,
        const QString &fileName) :
    m_documentHolder(documentHolder),
    m_interface(interface),
    m_fileName(fileName)
{}

/*!
 * Starts the process of extracting user bookmarks.
 */
void PdfDocumentUserBookmarksLoadTask::run()
{
    if (m_interface.isCanceled() || PdfTaskQueue::instance().blockedId().contains(m_documentHolder->id()))
        return;

    parseUserBookmarks(&m_bookmarks, m_fileName);
    m_interface.reportFinished(&m_bookmarks);
}

/*!
 * Cancels task.
 */
void PdfDocumentUserBookmarksLoadTask::cancel() {}

/*!
 * Returns document id.
 */
int PdfDocumentUserBookmarksLoadTask::id() const
{
    return m_documentHolder->id();
}
