// SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include <QStandardPaths>
#include <QFile>

#include "pdfdocumentholder.h"
#include "pdftaskqueue.h"

#include "pdfdocumentrenamebookmarksfiletask.h"

/*!
 * \class PdfDocumentRenameBookmarksFileTask
 * \brief The PdfDocumentRenameBookmarksFileTask class is a task for renaming bookmark.
 * \inmodule AmberPDF
 * \ingroup tasks
 */

/*!
 * Renames the file from the old (\a prevFileName) name to the new (\a newFileName) name.
 */
static bool renameBookmarksFile(const QString &prevFileName, const QString &newFileName)
{
    QString cachePath = QStandardPaths::writableLocation(QStandardPaths::CacheLocation);

    QString previousFilePath = cachePath + "/" + prevFileName;
    QString newFilePath = cachePath + "/" + newFileName;

    if (!QFile::exists(previousFilePath)) {
        return false;
    }

    // Refresh user bookmarks file if the new file with same id was created.
    if (QFile::exists(newFilePath)) {
        if (newFilePath != previousFilePath) {
            if (!QFile::remove(newFilePath)) {
                return false;
            }
        }
    }

    return QFile::rename(previousFilePath, newFilePath);
}

/*!
 * Parametrized constructor.
 * \a interface is QFutureInterface for future tasks.
 * \a document is QSharedPointer to PdfDocumentHolder object with document.
 * \a prevFileName is previous file name.
 * \a newFileName is new file name.
 */
PdfDocumentRenameBookmarksFileTask::PdfDocumentRenameBookmarksFileTask(const QFutureInterface<bool> &interface,
                                                                       const QSharedPointer<PdfDocumentHolder> &document,
                                                                       const QString &prevFileName,
                                                                       const QString &newFileName) :
    m_interface(interface),
    m_documentHolder(document),
    m_previousFileName(prevFileName),
    m_newFileName(newFileName)
{
}

/*!
 * Starts process of renaming bookmark.
 */
void PdfDocumentRenameBookmarksFileTask::run()
{
    if (m_interface.isCanceled() || PdfTaskQueue::instance().blockedId().contains(m_documentHolder->id()))
        return;

    bool result = renameBookmarksFile(m_previousFileName, m_newFileName);
    m_interface.reportFinished(&result);
}

/*!
 * Cancels task.
 */
void PdfDocumentRenameBookmarksFileTask::cancel() {}

/*!
 * Returns document id.
 */
int PdfDocumentRenameBookmarksFileTask::id() const
{
    return m_documentHolder->id();
}
