// SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include <QScopedPointer>
#include <QFile>

#include <pdfium/fpdf_doc.h>
#include <pdfium/fpdfview.h>

#include "pdfdocumentextractnewfileidtask.h"

/*!
 * \class PdfDocumentExtractNewFileIdTask
 * \brief The PdfDocumentExtractNewFileIdTask class is a task for extracting new file id.
 * \inmodule AmberPDF
 * \ingroup tasks
 */

/*!
 * Parametrized constructor.
 * \a path is path on device of new file.
 * \a interface is QFutureInterface for future tasks.
 */
PdfDocumentExtractNewFileIdTask::PdfDocumentExtractNewFileIdTask(const QString &path,
                                                                 const QFutureInterface<QString> &interface) :
    m_path(path), m_interface(interface)
{
}

/*!
 * Handles the process of extracting identifier from the file.
 */
void PdfDocumentExtractNewFileIdTask::run()
{
    FPDF_DOCUMENT newDocument = FPDF_LoadDocument(m_path.toUtf8().constData(), nullptr);

    if (!newDocument) {
        return;
    }

    QSharedPointer<fpdf_document_t__> document(newDocument);
    std::vector<char> buffer(256);

    unsigned long bufLen = FPDF_GetFileIdentifier(document.data(), FILEIDTYPE_CHANGING, buffer.data(), 256);

    if (bufLen == 0) {
        return;
    }

    QByteArray byteArray(buffer.data(), bufLen);

    // Check for the null byte.
    if (!byteArray.isEmpty() && byteArray[byteArray.size() - 1] == '\0') {
        byteArray.chop(1);
    }

    // Check if the byteArray has readable symbols.
    bool isReadable = true;
    for (char ch : byteArray) {
        if (!isprint(static_cast<unsigned char>(ch))) {
            isReadable = false;
            break;
        }
    }

    QString result;
    if (isReadable) {
        result = QString::fromLatin1(byteArray);
    } else {
        result = QString::fromLatin1(byteArray.toHex());
    }

    m_interface.reportFinished(&result);
}

/*!
 * Cancels the task.
 */
void PdfDocumentExtractNewFileIdTask::cancel() { }

/*!
 * Returns 0 as there is no document at the moment.
 */
int PdfDocumentExtractNewFileIdTask::id() const
{
    return 0;
}
