// SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include <QStandardPaths>
#include <QFile>
#include <QJsonDocument>
#include <QJsonObject>
#include <QDir>

#include "pdfdocumentholder.h"
#include "pdftaskqueue.h"

#include "pdfdocumentadduserbookmarktask.h"

/*!
 * \class PdfDocumentAddUserBookmarkTask
 * \brief The PdfDocumentAddUserBookmarkTask class is a task for adding new user bookmark.
 * \inmodule AmberPDF
 * \ingroup tasks
 */

/*!
 * Handles the process of adding new user bookmark to json file.
 */
static void writeNewBookmark(PdfDocumentAddUserBookmarkTask::NewBookmark &m_newBookmark) {
    QString cachePath = QStandardPaths::writableLocation(QStandardPaths::CacheLocation);
    QString filePath = cachePath + "/" + m_newBookmark.fileName;

    QDir dir(cachePath);
    if (!dir.exists()) {
        dir.mkpath(cachePath);
    }

    QFile file(filePath);

    QJsonObject rootObj;
    if (file.exists() && file.open(QIODevice::ReadOnly)) {
        QByteArray jsonData = file.readAll();
        file.close();

        QJsonDocument doc = QJsonDocument::fromJson(jsonData);
        if (doc.isObject()) {
            rootObj = doc.object();
        }
    }

    QJsonObject userBookmarksObj;
    if (rootObj.contains("bookmarks") && rootObj["bookmarks"].isObject()) {
        userBookmarksObj = rootObj["bookmarks"].toObject();
    }

    userBookmarksObj[QString::number(m_newBookmark.pageIndex)] = m_newBookmark.pageText;
    rootObj["bookmarks"] = userBookmarksObj;

    if (file.open(QIODevice::WriteOnly)) {
        QJsonDocument saveDoc(rootObj);
        file.write(QJsonDocument(rootObj).toJson(QJsonDocument::Indented));
        file.close();
    }
}

/*!
 * Parametrized constructor.
 * \a newBookmark is initialized object of NewBookmark type with parameters of annotation.
 * \a interface is QFutureInterface for future tasks.
 * \a document is QSharedPointer to PdfDocumentHolder object with document.
 */
PdfDocumentAddUserBookmarkTask::PdfDocumentAddUserBookmarkTask(
        const NewBookmark &newBookmark,
        const QFutureInterface<bool> &interface,
        const QSharedPointer<PdfDocumentHolder> &document) :
    m_newBookmark(newBookmark),
    m_interface(interface),
    m_documentHolder(document)
{
}

/*!
 * Starts  process of adding new user bookmark.
 */
void PdfDocumentAddUserBookmarkTask::run()
{
    if (m_interface.isCanceled() || PdfTaskQueue::instance().blockedId().contains(m_documentHolder->id()))
        return;

    writeNewBookmark(m_newBookmark);
    m_interface.reportFinished();
}

/*!
 * Cancels task.
 */
void PdfDocumentAddUserBookmarkTask::cancel() {}

/*!
 * Returns document id.
 */
int PdfDocumentAddUserBookmarkTask::id() const
{
    return m_documentHolder->id();
}
