/****************************************************************************
**
** Copyright (C) 2021 - 2022 Open Mobile Platform LLC.
** Contact: https://community.omprussia.ru/open-source
**
** This file is part of the AmberPDF project.
**
** $QT_BEGIN_LICENSE:BSD$
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Open Mobile Platform LLC copyright holder nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QtAlgorithms>
#include <pdfium/fpdf_text.h>

#include "pdftaskqueue.h"
#include "pdfdocumentholder.h"
#include "tasks/pdfpageclosetask.h"
#include "tasks/pdfpageannotationloadtask.h"
#include "tasks/pdfpagerendertask.h"
#include "tasks/pdfpagewordstask.h"
#include "tasks/pdfpagesizetask.h"
#include "tasks/pdfpageaddannotationtask.h"
#include "tasks/pdfpageremoveannotationtask.h"
#include "tasks/pdfpageaddinkannotationtask.h"
#include "tasks/pdfpagedrawsolidpaths.h"
#include "tasks/pdfpageaddimagetask.h"
#include "tasks/pdfpageremovesearchresultannotationstask.h"
#include "tasks/pdfpageaddinkannotationtask.h"

#include "pdfpage_p.h"
#include "pdfpage.h"

#define TEXT_FIND_X_TOLERANCE 50
#define TEXT_FIND_Y_TOLERANCE 50

/*!
 * \class PdfPageData
 * \brief The PdfPageData class contains all data members of PdfPage.
 * \inmodule AmberPDF
 */

/*!
 * Default constructor.
 */
PdfPageData::PdfPageData() : m_pageNumber(-1) { }

/*!
 * Parametrized constructor.
 */
PdfPageData::PdfPageData(int pageNumber, QSharedPointer<fpdf_page_t__> page,
                         QSharedPointer<PdfDocumentHolder> doc)
    : m_pageNumber(pageNumber), m_page(page), m_documentHolder(doc)
{
}

/*!
 * Copy constructor.
 */
PdfPageData::PdfPageData(const PdfPageData &other)
    : QSharedData(other),
      m_pageNumber(other.m_pageNumber),
      m_page(other.m_page),
      m_documentHolder(other.m_documentHolder),
      m_originalSize(other.m_originalSize)
{
}

/*!
 * Move constructor.
 */
PdfPageData::PdfPageData(PdfPageData &&other)
    : QSharedData(qMove(other)),
      m_pageNumber(qMove(other.m_pageNumber)),
      m_page(qMove(other.m_page)),
      m_documentHolder(qMove(other.m_documentHolder)),
      m_originalSize(qMove(other.m_originalSize))
{
}

/*!
 * Copy assignment operation for \a other.
 */
PdfPageData &PdfPageData::operator=(const PdfPageData &other)
{
    m_pageNumber = other.m_pageNumber;
    m_page = other.m_page;
    m_documentHolder = other.m_documentHolder;
    m_originalSize = other.m_originalSize;
    m_words = other.m_words;
    m_originalSize = other.m_originalSize;
    return *this;
}

/*!
 * Move assignment operation.
 */
PdfPageData &PdfPageData::operator=(PdfPageData &&other)
{
    qSwap(m_pageNumber, other.m_pageNumber);
    qSwap(m_page, other.m_page);
    qSwap(m_documentHolder, other.m_documentHolder);
    qSwap(m_originalSize, other.m_originalSize);
    qSwap(m_words, other.m_words);
    qSwap(m_originalSize, other.m_originalSize);
    return *this;
}

/*!
 * Destructor.
 */
PdfPageData::~PdfPageData()
{
    if (!m_originalSize.isFinished())
        m_originalSize.cancel();

    auto wordsFuture = m_words.future();
    if (wordsFuture.isFinished()) {
        auto wordsList = wordsFuture.result();
        QMutableListIterator<QObject *> wordsIt(wordsList);
        while (wordsIt.hasNext())
            delete wordsIt.next();
    } else {
        wordsFuture.cancel();
    }

    auto closePageInterface = QFutureInterface<void>();
    auto *closePageTask = new PdfPageCloseTask(m_documentHolder, closePageInterface, m_page);
    closePageTask->run();
    delete closePageTask;

    closePageInterface.reportStarted();
    closePageInterface.future().waitForFinished();
}

/*!
 * \class PdfPage
 * \brief The PdfPage class is a wrapper class of pdfium page.
 * \inmodule AmberPDF
 */

/*!
 * Default constructor.
 */
PdfPage::PdfPage() : d(new PdfPageData()) { }

/*!
 * Destructor.
 */
PdfPage::~PdfPage() = default;

/*!
 * Returns index of page in the document.
 */
int PdfPage::pageNumber() const
{
    return d->m_pageNumber;
}

/*!
 * Returns is page valid.
 */
bool PdfPage::isValid() const
{
    if (!d->m_documentHolder && !d->m_page)
        return false;

    return d->m_page;
}

/*!
 * Initialize a PdfPageSizeTask and returns its result in future.
 */
QFuture<QSizeF> PdfPage::originalSize()
{
    if (d->m_originalSize.isFinished() || d->m_originalSize.isRunning())
        return d->m_originalSize.future();

    auto task = QSharedPointer<PdfTask>(new PdfPageSizeTask(d->m_pageNumber, d->m_documentHolder, d->m_originalSize));
    auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                         d->m_documentHolder->id());
    if (!addingResult) {
        task.reset();
        return {};
    }

    d->m_originalSize.reportStarted();

    return d->m_originalSize.future();
}

/*!
 * Initialize a PdfPageRenderFullTask and returns its result in future.
 * Currently unused.
 */
QFuture<QImage> PdfPage::bitmapFull(qreal pageScale, int renderFlags) const
{
    if (!d->m_documentHolder || !d->m_page)
        return {};

    if (!d->m_originalSize.isFinished())
        return {};

    QFutureInterface<QImage> interface;
    const auto &future = interface.future();

    auto task =
            QSharedPointer<PdfTask>(new PdfPageRenderFullTask(pageScale, renderFlags, d->m_originalSize.future().result(),
                                      d->m_page, d->m_documentHolder, interface));
    auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::Low,
                                                         d->m_documentHolder->id());
    if (!addingResult) {
        task.reset();
        return {};
    }

    interface.reportStarted();

    return future;
}

/*!
 * Initialize a PdfPageRenderPartTask and returns its result in future.
 */
QFuture<QSharedPointer<QImage>> PdfPage::bitmapPart(qreal pageScaleX, qreal pageScaleY, int renderFlags, qreal zoom,
                                    const QPointF &bias) const
{
    if (!d->m_documentHolder || !d->m_page)
        return {};

    if (!d->m_originalSize.isFinished())
        return {};

    QFutureInterface<QSharedPointer<QImage>> interface;
    const auto &future = interface.future();

    auto task = QSharedPointer<PdfTask>(new PdfPageRenderPartTask(pageScaleX, pageScaleY, renderFlags, zoom, bias,
                                           d->m_originalSize.future().result(), d->m_pageNumber,
                                           d->m_documentHolder, interface));
    auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                         d->m_documentHolder->id());
    if (!addingResult) {
        task.reset();
        return {};
    }

    interface.reportStarted();

    return future;
}

/*!
 * Initialize a PdfPageAnnotationTask and returns its result in future.
 */
QFuture<QList<QSharedPointer<QObject>>> PdfPage::annotations()
{
    if (!d->m_documentHolder || !d->m_page)
        return {};

    QFutureInterface<QList<QSharedPointer<QObject>>> interface;
    const auto &future = interface.future();

    auto task = QSharedPointer<PdfTask>(new PdfPageAnnotationTask(d->m_page, QList<QSharedPointer<QObject>>(), interface,
                                           d->m_documentHolder));
    auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                         d->m_documentHolder->id());
    if (!addingResult) {
        task.reset();
        return {};
    }
    interface.reportStarted();

    return future;
}

/*!
 * Initialize a PdfPageWordsTask and returns its result in future.
 */
QFuture<QList<QObject *>> PdfPage::words()
{
    if (!d->m_documentHolder || !d->m_page)
        return {};
    auto origSizeF = originalSize();
    origSizeF.waitForFinished();
    auto origSize = origSizeF.result();
    auto task =
            QSharedPointer<PdfTask>(new PdfPageWordsTask(d->m_page, QList<QObject *>(), origSize,
                                 d->m_words, d->m_documentHolder));
    auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                         d->m_documentHolder->id());
    if (!addingResult) {
        task.reset();
        return {};
    }

    d->m_words.reportStarted();
    return d->m_words.future();
}

/*!
 * Initialize a PdfPageAddAnnotationTask and returns its result in future.
 */
QFuture<bool> PdfPage::addAnnotation(const QRectF &rect, const QColor &color, const QString &author,
                                     const QString &content, const QString &annotationType,
                                     const QList<QRectF> &attachedPoints)
{
    if (!d->m_documentHolder || !d->m_page)
        return {};

    QFutureInterface<bool> interface;
    const auto &future = interface.future();

    PdfPageAddAnnotationTask::NewAnnotation newAnnotation{ rect, color, author, content,
                                                           annotationType, attachedPoints };

    auto task =
            QSharedPointer<PdfTask>(new PdfPageAddAnnotationTask(d->m_page, newAnnotation, interface, d->m_documentHolder));
    auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                         d->m_documentHolder->id());
    if (!addingResult) {
        task.reset();
        return {};
    }
    interface.reportStarted();
    return future;
}

/*!
 * Initialize a PdfPageAddInkAnnotationTask and returns its result in future.
 */
QFuture<bool> PdfPage::drawInkAnnotation(const QRectF &rect, const QColor &color, const float penSize,
                                              const QList<QList<QPointF>> &points,
                                              const QString &author,
                                              const QString &content, const int annotationType)
{
    if (!d->m_documentHolder || !d->m_page)
        return {};

    QFutureInterface<bool> interface;
    const auto &future = interface.future();

    PdfPageAddInkAnnotationTask::InkAnnotation newAnnotation{ rect, color, author, content, points, penSize };

    auto task = QSharedPointer<PdfTask>(new PdfPageAddInkAnnotationTask(d->m_page, newAnnotation, interface,
                                              d->m_documentHolder));
    auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                         d->m_documentHolder->id());
    if (!addingResult) {
        task.reset();
        return {};
    }
    interface.reportStarted();
    return future;

    if (annotationType == Highlight) {
        PdfPageAddAnnotationTask::NewAnnotation newAnnotation{ rect, color, author, content, "Highlight", {} };

        auto task = QSharedPointer<PdfTask>(new PdfPageAddAnnotationTask(d->m_page, newAnnotation, interface,
                                                  d->m_documentHolder));
        auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                             d->m_documentHolder->id());
        if (!addingResult) {
            task.reset();
            return {};
        }
    } else if (annotationType == Ink) {
        QList<QPointF> pointfs;
        QList<QList<QPointF>> pointflist;
        for (int i = 0; i < points.first().size() ; i++) {
            pointfs.append(QPointF(points.first().at(i).x(), points.first().at(i).y()));
        }
        pointflist.append(pointfs);
        PdfPageDrawSolidPaths::Strokes strokes{ rect, color, points, static_cast<int>(penSize) };

        auto task = QSharedPointer<PdfTask>(new PdfPageDrawSolidPaths(d->m_page, strokes, interface, d->m_documentHolder));
        auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                             d->m_documentHolder->id());
        if (!addingResult) {
            task.reset();
            return {};
        }
    }

    interface.reportStarted();
    return future;
}

/*!
 * Initialize a PdfPageRemoveAnnotationTask and returns its result in future.
 */
QFuture<bool> PdfPage::removeAnnotation(int annotationIndex)
{
    if (!d->m_documentHolder || !d->m_page)
        return {};

    QFutureInterface<bool> interface;
    const auto &future = interface.future();

    auto task = QSharedPointer<PdfTask>(new PdfPageRemoveAnnotationTask(d->m_page, annotationIndex, interface,
                                                 d->m_documentHolder));
    auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                         d->m_documentHolder->id());
    if (!addingResult) {
        task.reset();
        return {};
    }

    interface.reportStarted();
    return future;
}

/*!
 * Initialize a PdfPageRemoveSearchResultAnnotationsTask and returns its result in future.
 */
QFuture<bool> PdfPage::removeTextSearchResult()
{
    if (!d->m_documentHolder || !d->m_page)
        return {};

    QFutureInterface<bool> interface;
    const auto &future = interface.future();

    auto task = QSharedPointer<PdfTask>(new PdfPageRemoveSearchResultAnnotationsTask(d->m_page, interface,
                                                 d->m_documentHolder));
    auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                         d->m_documentHolder->id());
    if (!addingResult) {
        task.reset();
        return {};
    }

    interface.reportStarted();
    return future;
}

/*!
 * Initialize a PdfPageAddImageTask and returns its result in future.
 */
QFuture<bool> PdfPage::addImage(const QPointF &topLeft, const qreal &pageRate,
                                const QString &imagePath)
{
    if (!d->m_documentHolder || !d->m_page)
        return {};

    QFutureInterface<bool> interface;
    const auto &future = interface.future();

    auto task = QSharedPointer<PdfTask>(new PdfPageAddImageTask(d->m_page, imagePath, topLeft, pageRate, interface,
                                         d->m_documentHolder));
    auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                         d->m_documentHolder->id());
    if (!addingResult) {
        task.reset();
        return {};
    }

    interface.reportStarted();
    return future;
}

/*!
 * Constructs rectangle for addAnnotation method.
 */
QFuture<bool> PdfPage::highlightText(QPoint startPoint, QPoint endPoint, QColor color)
{
    QList<QRectF> attachedPoints = findTextRects(startPoint, endPoint);
    if (attachedPoints.isEmpty())
        return {};

    QRectF globalRect = attachedPoints.at(0);

    for (int i = 1; i < attachedPoints.size(); i++) {
        globalRect.setLeft(qMin(attachedPoints[i].left(), globalRect.left()));
        globalRect.setRight(qMax(attachedPoints[i].right(), globalRect.right()));
        globalRect.setTop(qMin(attachedPoints[i].top(), globalRect.top()));
        globalRect.setBottom(qMax(attachedPoints[i].bottom(), globalRect.bottom()));
    }

    return addAnnotation(globalRect, color, "defaultuser", "", "HighLight", attachedPoints);
}

/*!
 * Finds rectangles by QPoints.
 */
QList<QRectF> PdfPage::findTextRects(QPoint startPoint, QPoint endPoint)
{
    auto *textPage = FPDFText_LoadPage(d->m_page.data());
    if (!textPage) {
        return {};
    }

    double pageHeight = FPDF_GetPageHeight(d->m_page.data());
    double pageWidth = FPDF_GetPageWidth(d->m_page.data());
    int startIndex = FPDFText_GetCharIndexAtPos(textPage, startPoint.x(), pageHeight - startPoint.y(), TEXT_FIND_X_TOLERANCE, TEXT_FIND_Y_TOLERANCE);
    int endIndex = FPDFText_GetCharIndexAtPos(textPage, endPoint.x(), pageHeight - endPoint.y(), pageWidth, pageHeight);

    if (startIndex < 0)
        return {};
    if (endIndex < 0)
        return {};

    if (startIndex > endIndex)
        qSwap(startIndex, endIndex);

    QRectF attachedRect;
    double left, right, top, bottom;
    double maxTop = 0.0;
    double minBottom = pageHeight;
    bool isFirstRowChar = true;
    QString str;
    QList<QRectF> attachedPoints;

    for (int i = startIndex; i < endIndex; i++) {
        FPDFText_GetCharBox(textPage, i, &left, &right, &bottom, &top);
        char sym = FPDFText_GetUnicode(textPage, i);

        maxTop = qMax(maxTop, top);
        minBottom = qMin(minBottom, bottom);

        if (sym == '\n' || i == endIndex - 1) {
            attachedRect.setRight(right);
            attachedRect.setBottom(maxTop);
            attachedRect.setTop(minBottom);
            attachedRect.setHeight(maxTop - minBottom);
            attachedPoints.append(attachedRect);
            isFirstRowChar = true;
        } else if (isFirstRowChar) {
            attachedRect.setLeft(left);
            isFirstRowChar = false;
            maxTop = 0.0;
            minBottom = pageHeight;
        }

        str += sym;
    }
    return attachedPoints;
}

const QString PdfPage::text() const
{
    return d->text;
}
