/****************************************************************************
**
** Copyright (C) 2021 - 2025 Open Mobile Platform LLC.
** Contact: https://community.omprussia.ru/open-source
**
** This file is part of the AmberPDF project.
**
** $QT_BEGIN_LICENSE:BSD$
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Open Mobile Platform LLC copyright holder nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QFileInfo>
#include <QFutureWatcher>
#include <pdfpagewordstask.h>
#include "pdftaskqueue.h"
#include "tasks/pdfdocumentloadtask.h"
#include "tasks/pdfdocumentloadfrommemtask.h"
#include "tasks/pdfdocumentpagesizetask.h"
#include "tasks/pdfdocumentpagecounttask.h"
#include "tasks/pdfpageloadtask.h"
#include "tasks/pdfdocumentbookmarksloadtask.h"
#include "tasks/pdfdocumentuserbookmarksloadtask.h"
#include "tasks/pdfdocumentdeleteuserbookmarktask.h"
#include "tasks/pdfdocumentadduserbookmarktask.h"
#include "tasks/pdfdocumentidloadtask.h"
#include "tasks/pdfdocumentrenamebookmarksfiletask.h"
#include "tasks/pdfdocumentextractnewfileidtask.h"
#include "tasks/pdfdocumentversiongettask.h"
#include "tasks/pdfdocumentsavetask.h"
#include "tasks/pdfdocumentexporttask.h"
#include "tasks/pdfdocumentphrasesearchtask.h"
#include "pdfdocument.h"
#include "pdfword.h"
#include "pdfpage.h"
#include "pdfdocumentholder.h"
#include "pdfdocument_p.h"

/*!
 * Clear references of document data.
 */
void PdfDocumentPrivate::clear()
{
    if (m_documentHolder) {
        PdfTaskQueue::instance().addBlockedId(m_documentHolder->id());

        if (!m_openPages.empty()) {
            for (auto &future : m_openPages) {
                future.cancel();
                if (!future.isCanceled())
                    future.result().clear();
            }
            m_openPages.clear();
        }
    }

    m_documentHolder.clear();
    m_documentHolder.reset();
    m_status = PdfDocument::DocumentStatus::Unknown;
}

/*!
 * Handle loaded document.
 */
void PdfDocumentPrivate::handleDocument(QPair<int, QSharedPointer<fpdf_document_t__>> loadResult)
{
    m_status = loadResult.second  ? PdfDocument::DocumentStatus::Success
                                     : static_cast<PdfDocument::DocumentStatus>(loadResult.first);

    if (m_status != PdfDocument::DocumentStatus::Success)
        return;

    m_documentHolder.clear();
    m_documentHolder = QSharedPointer<PdfDocumentHolder>(new PdfDocumentHolder(loadResult.second));
    if (!m_documentHolder) {
        m_status = PdfDocument::DocumentStatus::InternalError;
        return;
    }

    m_status = PdfDocument::DocumentStatus::Success;
}

/*!
 * \class PdfDocument
 * \brief The PdfDocument class is a wrapper class of pdfium document.
 * \inmodule AmberPDF
 */

/*!
 * Default constructor.
 */
PdfDocument::PdfDocument(QObject *parent) : QObject(parent),
    d_ptr(new PdfDocumentPrivate())
{
    connect(this, &PdfDocument::statusChanged, this, [&](DocumentStatus status) {
        if (status != DocumentStatus::Success)
            return;

        Q_D(PdfDocument);
        QFutureInterface<int> pageCountInterface;
        d->m_pageCountWatcher.setFuture(pageCountInterface.future());
        auto pageCountTask = QSharedPointer<PdfTask>(new PdfDocumentPageCountTask(d->m_documentHolder, pageCountInterface));
        auto addingResult = PdfTaskQueue::instance().addTask(pageCountTask, PdfTaskQueue::TaskPriority::High, d->m_documentHolder->id());
        if (!addingResult) {
            pageCountTask.reset();
            d->m_status = DocumentStatus::InternalError;
            emit statusChanged(d->m_status);
        }

        QFutureInterface<int> versionInterface;
        d->m_versionWatcher.setFuture(versionInterface.future());
        auto versionTask = QSharedPointer<PdfTask>(new PdfDocumentVersionGetTask(d->m_documentHolder, versionInterface));
        addingResult = PdfTaskQueue::instance().addTask(versionTask, PdfTaskQueue::TaskPriority::High, d->m_documentHolder->id());
        if (!addingResult) {
            versionTask.reset();
            d->m_status = DocumentStatus::InternalError;
            emit statusChanged(d->m_status);
        }
    });

    connect(&d_ptr->m_pageCountWatcher, &QFutureWatcher<int>::finished, this, [&]() {
        Q_D(PdfDocument);
        if (d->m_pageCountWatcher.isCanceled())
            return;

        auto pageCount = d->m_pageCountWatcher.result();
        d_ptr->m_pagesCount = pageCount;
        emit pageCountChanged(d_ptr->m_pagesCount);
    });

    connect(&d_ptr->m_versionWatcher, &QFutureWatcher<int>::finished, this, [&]() {
        Q_D(PdfDocument);
        if (d->m_versionWatcher.isCanceled())
            return;

        d_ptr->m_fileVersion = d->m_versionWatcher.result();
        emit fileVersionChanged(d_ptr->m_fileVersion);
    });
}

/*!
 * Destructor.
 */
PdfDocument::~PdfDocument()
{
    Q_D(PdfDocument);
    d->clear();

    delete d_ptr;
}

/*!
 * Returns current status of document.
 */
PdfDocument::DocumentStatus PdfDocument::status() const
{
    Q_D(const PdfDocument);
    return d->m_status;
}

/*!
 * Returns page count of document.
 */
int PdfDocument::pageCount() const
{
    Q_D(const PdfDocument);

    if (!d->m_documentHolder)
        return -1;

    if (d->m_pageCountWatcher.isFinished() && !d->m_pageCountWatcher.isCanceled())
        return d->m_pageCountWatcher.result();

    return -1;
}

/*!
 * Returns path to document.
 */
QString PdfDocument::path() const
{
    Q_D(const PdfDocument);
    return d->m_path;
}

/*!
 * Initialize a PdfPageLoadTask and returns its result in future.
 */
QFuture<QSharedPointer<PdfPage>> PdfDocument::page(int pageIndex) const
{
    Q_D(const PdfDocument);

    if (pageIndex < 0 || pageIndex > pageCount())
        return {  };

    if (!d->m_documentHolder || status() != DocumentStatus::Success)
        return {  };

    if (!d->m_openPages.contains(pageIndex)) {
        QFutureInterface<QSharedPointer<PdfPage>> interface;

        auto loadPageTask = QSharedPointer<PdfTask>(new PdfPageLoadTask(d->m_documentHolder, interface, pageIndex));
        auto addingResult = PdfTaskQueue::instance().addTask(loadPageTask, PdfTaskQueue::TaskPriority::Low, d->m_documentHolder->id());
        if (!addingResult) {
            loadPageTask.reset();
            return {  };
        }

        interface.reportStarted();

        d->m_openPages.insert(pageIndex, interface.future());
    }

    return d->m_openPages.value(pageIndex);
}

/*!
 * Initialize a PdfDocumentPageSizeTask and returns its result in future.
 */
QFuture<QSizeF> PdfDocument::pageSize(int pageIndex) const
{
    Q_D(const PdfDocument);

    if (pageIndex < 0 || pageIndex > pageCount())
        return {  };

    if (!d->m_documentHolder || status() != DocumentStatus::Success)
        return {  };

    QFutureInterface<QSizeF> interface;

    auto loadPageTask = QSharedPointer<PdfTask>(new PdfDocumentPageSizeTask(d->m_documentHolder, interface, pageIndex));
    auto addingResult = PdfTaskQueue::instance().addTask(loadPageTask, PdfTaskQueue::TaskPriority::High, d->m_documentHolder->id());
    if (!addingResult) {
        loadPageTask.reset();
        return {  };
    }

    interface.reportStarted();

    return interface.future();
}

/*!
 * \fn PdfDocument::loadDocument(const QString &fileName, const QString &password)
 *
 * Loads document from PDF file.
 * \a fileName is path to file on device.
 * \a password is password for document loading if needed.
 */
void PdfDocument::loadDocument(const QString &fileName, const QString &password)
{
    Q_D(PdfDocument);

    if (d->m_status == DocumentStatus::Success)
        d->clear();

    d->m_status = DocumentStatus::Loading;
    emit statusChanged(d->m_status);

    d->m_path = fileName;
    emit pathChanged(d->m_path);

    QFileInfo fileInfo(fileName);
    if (!fileInfo.isReadable()) {
        d->m_status = DocumentStatus::FileError;
        emit statusChanged(d->m_status);
        return;
    }

    auto pdfiumLoadFuture = d->fromPdf(fileName, password);
    if (pdfiumLoadFuture.isFinished()) {
        d->handleDocument(pdfiumLoadFuture.result());
        emit statusChanged(d->m_status);
        return;
    }

    auto pdfLoadWatcher = new QFutureWatcher<QPair<int, QSharedPointer<fpdf_document_t__>>>(this);
    pdfLoadWatcher->setFuture(pdfiumLoadFuture);

    connect(pdfLoadWatcher, &QFutureWatcher<QPair<int, QSharedPointer<fpdf_document_t__>>>::finished, this, [&]() {
        Q_D(PdfDocument);
        auto watcher = dynamic_cast<QFutureWatcher<QPair<int, QSharedPointer<fpdf_document_t__>>>*>(sender());
        if (watcher == nullptr)
            return;
        if (watcher->isCanceled())
            d->m_status = DocumentStatus::InternalError;
        else
            d->handleDocument(watcher->result());

        emit statusChanged(d->m_status);
        watcher->deleteLater();
    });
}

/*!
 * \fn PdfDocument::loadDocument(const QByteArray &buffer, const QString &password)
 *
 * Loads document from memory.
 * \a buffer is memory buffer.
 * \a password is password for document loading if needed.
 */
void PdfDocument::loadDocument(const QByteArray &buffer, const QString &password)
{
    Q_D(PdfDocument);

    if (d->m_status == DocumentStatus::Success)
        d->clear();

    d->m_status = DocumentStatus::Loading;
    emit statusChanged(d->m_status);

    auto pdfiumLoadFuture = d->fromMem(buffer, password);
    if (pdfiumLoadFuture.isFinished()) {
        d->handleDocument(pdfiumLoadFuture.result());
        emit statusChanged(d->m_status);
        return;
    }

    auto pdfLoadWatcher = new QFutureWatcher<QPair<int, QSharedPointer<fpdf_document_t__>>>(this);
    pdfLoadWatcher->setFuture(pdfiumLoadFuture);

    connect(pdfLoadWatcher, &QFutureWatcher<QPair<int, QSharedPointer<fpdf_document_t__>>>::finished, this, [&]() {
        Q_D(PdfDocument);
        auto watcher = dynamic_cast<QFutureWatcher<QPair<int, QSharedPointer<fpdf_document_t__>>>*>(sender());
        if (watcher == nullptr)
            return;

        if (watcher->isCanceled())
            d->m_status = DocumentStatus::InternalError;
        else
            d->handleDocument(watcher->result());
        emit statusChanged(d->m_status);
        watcher->deleteLater();
    });
}

/*!
 * Initialize a PdfDocumentBookmarksLoadTask and returns its result in future.
 */
QFuture<QVector<PdfBookmark>> PdfDocument::bookmarks() const
{
    Q_D(const PdfDocument);
    QFutureInterface<QVector<PdfBookmark>> interface;

    auto task = QSharedPointer<PdfTask>(new PdfDocumentBookmarksLoadTask(d->m_documentHolder, interface));

    auto addResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High);
    if (!addResult) {
        task.reset();
        return {  };
    }

    interface.reportStarted();

    return interface.future();
}

/*!
 * Initialize a PdfDocumentUserBookmarksLoadTask and returns its result in future.
 */
QFuture<QVector<UserPdfBookmark>> PdfDocument::userBookmarks(const QString &fileName) const
{
    Q_D(const PdfDocument);
    QFutureInterface<QVector<UserPdfBookmark>> interface;

    auto task = QSharedPointer<PdfTask>(new PdfDocumentUserBookmarksLoadTask(d->m_documentHolder, interface, fileName));

    auto addResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High);
    if (!addResult) {
        task.reset();
        return {  };
    }

    interface.reportStarted();

    return interface.future();
}

/*!
 * Initialize a PdfDocumentAddUserBookmarkTask and returns its result in future.
 */
QFuture<bool> PdfDocument::addUserBookmark(int pageNumber, const QString &pageText, const QString &fileName)
{
    Q_D(const PdfDocument);
    QFutureInterface<bool> interface;
    const auto &future = interface.future();

    PdfDocumentAddUserBookmarkTask::NewBookmark newBookmark{pageNumber, pageText, fileName};

    auto task =
            QSharedPointer<PdfTask>(new PdfDocumentAddUserBookmarkTask(newBookmark, interface, d->m_documentHolder));
    auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                         d->m_documentHolder->id());
    if (!addingResult) {
        task.reset();
        return {};
    }

    interface.reportStarted();

    return future;
}

/*!
 * Initialize a PdfDocumentDeleteUserBookmarkTask and returns its result in future.
 */
QFuture<bool> PdfDocument::deleteUserBookmark(int pageNumber, const QString &fileName)
{
    Q_D(const PdfDocument);
    QFutureInterface<bool> interface;
    const auto &future = interface.future();

    auto task = QSharedPointer<PdfTask>(new PdfDocumentDeleteUserBookmarkTask(pageNumber, fileName, interface, d->m_documentHolder));
    auto addingResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                         d->m_documentHolder->id());
    if (!addingResult) {
        task.reset();
        return {};
    }

    interface.reportStarted();
    return future;
}

/*!
 * Initialize a PdfDocumentIdLoadTask and returns its result in future.
 */
QFuture<QString> PdfDocument::getFileId() const
{
    Q_D(const PdfDocument);
    QFutureInterface<QString> interface;

    auto task = QSharedPointer<PdfTask>(new PdfDocumentIdLoadTask(d->m_documentHolder, interface));

    auto addResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High);
    if (!addResult) {
        task.reset();
        return { };
    }

    interface.reportStarted();
    return interface.future();
}


/*!
 * Initialize a PdfDocumentExtractNewFileIdTask and returns its result in future.
 */
QFuture<QString> PdfDocument::getNewFileId(const QString &filePath) const
{
    QFutureInterface<QString> interface;
    auto task = QSharedPointer<PdfTask>(new PdfDocumentExtractNewFileIdTask(filePath, interface));

    auto addResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High);
    if (!addResult) {
        task.reset();
        return { };
    }

    interface.reportStarted();
    return interface.future();
}

/*!
 * Initialize a PdfDocumentRenameBookmarksFileTask and returns its result in future.
 */
QFuture<bool> PdfDocument::renameBookmarksFile(const QString &prevFileName, const QString &newFileName) const
{
    Q_D(const PdfDocument);
    QFutureInterface<bool> interface;

    auto task =
            QSharedPointer<PdfTask>(new PdfDocumentRenameBookmarksFileTask(interface, d->m_documentHolder, prevFileName, newFileName));
    auto renameResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High,
                                                         d->m_documentHolder->id());
    if (!renameResult) {
        task.reset();
        return {};
    }

    interface.reportStarted();
    return interface.future();
}

/*!
 * Initialize a PdfDocumentRenameBookmarksFileTask and returns its result in future.
 */
int PdfDocument::fileVersion() const
{
    Q_D(const PdfDocument);
    return d->m_fileVersion;
}

/*!
 * Initialize a PdfDocumentSaveTask and returns its result in future.
 */
bool PdfDocument::saveDocumentAs(const QString &path) const
{
    Q_D(const PdfDocument);
    QFutureInterface<bool> interface;
    
    auto task = QSharedPointer<PdfTask>(new PdfDocumentSaveTask(path, interface, d->m_documentHolder->document()));

    auto addResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::Immediately);
    if (!addResult) {
        task.reset();
        return false;
    }

    interface.reportStarted();

    return interface.future().result();
}

/*!
 * Initialize a PdfDocumentExportTask and returns its result in future.
 */
bool PdfDocument::exportDocument(QIODevice *output) const
{
    Q_D(const PdfDocument);
    QFutureInterface<bool> interface;

    auto task = QSharedPointer<PdfTask>(new PdfDocumentExportTask(output, interface, d->m_documentHolder->document()));

    auto addResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::Immediately);
    if (!addResult) {
        task.reset();
        return false;
    }

    interface.reportStarted();

    return interface.future().result();
}

/*!
 * Initialize a PdfDocumentPhraseSearchTask and returns its result in future.
 */
QFuture<PdfPageTexts*> PdfDocument::searchPhrase(const QString &phrase, PdfPageTexts* pageTextsModel) const
{
    Q_D(const PdfDocument);

    QFutureInterface<PdfPageTexts*> interface;
    auto task = QSharedPointer<PdfTask>(new PdfDocumentPhraseSearchTask(phrase, pageTextsModel, interface, d->m_documentHolder));

    auto addResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::Immediately);
    if (!addResult) {
        task.reset();
        return {};
    }

    interface.reportStarted();

    return interface.future();
}
/*!
 * Initialize a PdfDocumentLoadTask and returns its result in future.
 */
QFuture<QPair<int, QSharedPointer<fpdf_document_t__>>> PdfDocumentPrivate::fromPdf(const QString &fileName, const QString &password)
{
    QFutureInterface<QPair<int, QSharedPointer<fpdf_document_t__>>> interface;
    auto task = QSharedPointer<PdfTask>(new PdfDocumentLoadTask(fileName, password, interface));

    auto addResult = PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High);

    if (!addResult)
        task.reset();

    interface.reportStarted();

    return interface.future();
}

/*!
 * Initialize a PdfDocumentLoadFromMemTask and returns its result in future.
 */
QFuture<QPair<int, QSharedPointer<fpdf_document_t__>>> PdfDocumentPrivate::fromMem(const QByteArray &buffer, const QString &password)
{
    QFutureInterface<QPair<int, QSharedPointer<fpdf_document_t__>>> interface;
    auto task = QSharedPointer<PdfTask>(new PdfDocumentLoadFromMemTask(buffer, password, interface));

    PdfTaskQueue::instance().addTask(task, PdfTaskQueue::TaskPriority::High);
    interface.reportStarted();

    return interface.future();
}
