// SPDX-FileCopyrightText: 2022 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include "filesmodel.h"
#include "sortmodel.h"

SortModel::SortModel(QObject *parent) : QSortFilterProxyModel(parent)
{
    this->setFilterRole(FilesModel::FilePropertyRoles::FileNameRole);
    setSortParameter(SortParameter::Date);
    setSortCaseSensitivity(Qt::CaseInsensitive);
}

SortModel::~SortModel() = default;

FilesModel *SortModel::sourceModel() const
{
    return static_cast<FilesModel *>(QSortFilterProxyModel::sourceModel());
}

int SortModel::sortParameter() const
{
    return m_sortParameter;
}

bool SortModel::ascendingOrder() const
{
    return m_ascendingOrder;
}

bool SortModel::removeFile(const QString &fileName)
{
    if (sourceModel() == nullptr)
        return false;

    auto filesModel = qobject_cast<FilesModel *>(sourceModel());
    if (filesModel == nullptr)
        return false;

    return filesModel->removeFile(fileName);
}

void SortModel::setSourceModel(FilesModel *sourceModel)
{
    QSortFilterProxyModel::setSourceModel(static_cast<QAbstractItemModel *>(sourceModel));
}

void SortModel::setSortParameter(int sortParameter)
{
    if (m_sortParameter == sortParameter)
        return;

    m_sortParameter = sortParameter;
    const auto order = m_ascendingOrder ? Qt::AscendingOrder : Qt::DescendingOrder;

    switch (m_sortParameter) {
    case None:
        break;
    case FileName:
        setSortRole(FilesModel::FilePropertyRoles::FileNameRole);
        break;
    case Size:
        setSortRole(FilesModel::FilePropertyRoles::SizeRole);
        break;
    case Date:
        setSortRole(FilesModel::FilePropertyRoles::LastChangesRole);
        break;
    default:
        break;
    }

    emit sortParameterChanged(m_sortParameter);

    sort(0, order);
}

void SortModel::setAscendingOrder(bool ascendingOrder)
{
    if (m_ascendingOrder == ascendingOrder)
        return;

    m_ascendingOrder = ascendingOrder;
    const auto order = m_ascendingOrder ? Qt::AscendingOrder : Qt::DescendingOrder;

    emit ascendingOrderChanged();

    sort(0, order);
}
