// SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Share 1.0
import Sailfish.Pickers 1.0
import ru.auroraos.PdfViewer 1.0
import Aurora.Controls 1.0

Page {
    id: root

    property real timeFormat: Format.Timepoint

    readonly property var statesNames: ["loading", "work", "documentsNotFound"]
    property SortModel filesModel: SortModel {
        function escapeRegExp(text) {
            return text.replace(/[-[\]{}()*+?.,\\^$|#\s]/g, '\\$&');
        }

        objectName: "sortModel"
        sourceModel: FilesModel {
            onGraphChanged: {
                root.state = documentsCount > 0 ? root.statesNames[1] : root.statesNames[2];
                if (pageStack.currentPage === null || pageStack.currentPage.objectName === root.objectName)
                    return;
                var contentPage = pageStack.find(function (page) {
                        if (page.objectName === "contentPage")
                            return page;
                    });
                if (contentPage)
                    contentPage.checkCurrentFile();
            }
            onDataChanged: {
                root.timeFormat = "";
                root.timeFormat = Format.Timepoint;
            }

            Component.onCompleted: {
                root.state = statesNames[0];
                reset();
            }

            objectName: "filesModel"
        }
        filterRegExp: RegExp(escapeRegExp(filesView.searchText), "i")
    }

    function chooseFile(path) {
        while (pageStack.depth > 1) {
            pageStack.pop(undefined, PageStackAction.Immediate);
        }
        pageStack.push(Qt.resolvedUrl("ContentPage.qml"), {
                           "pdfPath": path === undefined ? "" : path
                       });
    }

    function setSearchMode(mode) {
        filesView.searchEnabled = mode;
        topBar.visible = !mode;
    }

    function defaultState() {
        setSearchMode(false);
        filesModel.setSortParameter(SortModel.FileName);
        filesModel.setAscendingOrder(true);
    }

    onStatusChanged: {
        if (status === PageStatus.Active) {
            defaultState();
        }
    }

    objectName: "filesPage"
    allowedOrientations: Orientation.All

    BusyIndicator {
        id: documentListLoadIndicator

        objectName: "busyIndicator"
        anchors.centerIn: parent
        size: BusyIndicatorSize.Large
    }

    Label {
        id: documentsNotFoundLabel

        anchors {
            verticalCenter: parent.verticalCenter
            horizontalCenter: parent.horizontalCenter
        }
        objectName: "errorLabel"
        enabled: filesView.count <= 0
        visible: enabled
        color: Theme.highlightColor
        textFormat: Text.PlainText
        text: qsTr("Documents not found")
        font.pixelSize: Theme.fontSizeExtraLarge
    }

    AppBar {
        id: topBar

        headerText: qsTr("PDF Viewer")

        AppBarSpacer {}

        AppBarButton {
            id: topBarSortBtn

            icon.source: filesModel.ascendingOrder ? "image://theme/icon-m-sort-asc" : "image://theme/icon-m-sort-desc"

            onClicked: {
                const order = filesModel.ascendingOrder;
                filesModel.setAscendingOrder(!order);
            }
        }

        AppBarButton {
            id: topBarSortModeBtn

            icon.source: "image://theme/icon-m-sortby"

            onClicked: sortModePopup.open()

            PopupMenu {
                id: sortModePopup

                PopupMenuItem {
                    text: qsTr("Name")

                    onClicked: {
                        filesModel.setSortParameter(SortModel.FileName);
                    }
                }
                PopupMenuItem {
                    text: qsTr("Size")

                    onClicked: {
                        filesModel.setSortParameter(SortModel.Size);
                    }
                }
                PopupMenuItem {
                    text: qsTr("Date")

                    onClicked: {
                        filesModel.setSortParameter(SortModel.Date);
                    }
                }
            }
        }

        AppBarButton {
            id: topBarAboutBtn

            icon.source: "image://theme/icon-m-more"

            onClicked: morePopup.open()

            PopupMenu {
                id: morePopup

                PopupMenuItem {
                    text: qsTr("Search file")

                    onClicked: {
                        setSearchMode(true);
                    }
                }
                PopupMenuItem {
                    text: qsTr("Select file")

                    onClicked: {
                        pageStack.push(filePicker);
                    }
                }
                PopupMenuItem {
                    text: qsTr("About")

                    onClicked: {
                        pageStack.push(Qt.resolvedUrl("AboutPage.qml"));
                    }
                }
            }
        }
    }

    Row {
        id: searchRowLayout

        width: parent.width

        spacing: Theme.paddingMedium

        IconButton {
            id: closeSearchBtn

            icon.source: "image://theme/icon-m-back"

            onClicked: {
                setSearchMode(false);
            }
        }

        SearchField {
            id: searchField

            onTextChanged: filesView.searchText = text
            onActiveChanged: if (active)
                forceActiveFocus()
            onHeightChanged: filesView.contentY -= height
            EnterKey.onClicked: focus = false

            objectName: "searchField"
            active: filesView.searchEnabled
            width: parent.width - Theme.paddingMedium - closeSearchBtn.width
            height: active ? Theme.itemSizeMedium : 0
            placeholderText: qsTr("Search documents")
            transitionDuration: 400
            inputMethodHints: Qt.ImhNoAutoUppercase | Qt.ImhPreferLowercase | Qt.ImhNoPredictiveText
        }
    }

    SilicaListView {
        id: filesView

        anchors.topMargin: topBar.visible ? topBar.height : Theme.itemSizeMedium

        property bool searchEnabled
        property string searchText

        onCountChanged: root.state = filesView.count > 0 ? root.statesNames[1] : root.statesNames[2]

        Component.onCompleted: {
            currentIndex = -1;
            defaultState();
        }

        objectName: "filesView"
        anchors.fill: parent
        delegate: fileDelegateComponent
        model: root.filesModel
    }

    Component {
        id: fileDelegateComponent

        ListItem {
            id: fileDelegate

            onClicked: root.chooseFile(model.path)

            objectName: "delegateItem"
            contentHeight: Theme.itemSizeMedium

            Icon {
                id: fileIcon

                objectName: "fileIcon"
                anchors {
                    left: parent.left
                    leftMargin: Theme.horizontalPageMargin
                    verticalCenter: parent.verticalCenter
                }
                source: "image://theme/icon-m-file-pdf-light"
            }

            Column {
                objectName: "delegateColumn"
                anchors {
                    left: fileIcon.right
                    leftMargin: Theme.paddingMedium
                    right: parent.right
                    rightMargin: Theme.horizontalPageMargin
                    verticalCenter: parent.verticalCenter
                }

                Label {
                    objectName: "filenameLabel"
                    width: parent.width
                    color: fileDelegate.highlighted ? Theme.highlightColor : Theme.primaryColor
                    text: filesView.searchText.length > 0 ? Theme.highlightText(model.fileName, filesView.searchText, Theme.highlightColor) : model.fileName
                    textFormat: filesView.searchText.length > 0 ? Text.StyledText : Text.PlainText
                    font.pixelSize: Theme.fontSizeMedium
                    truncationMode: TruncationMode.Fade
                }

                Item {
                    objectName: "fileInfoItem"
                    width: parent.width
                    height: fileSizeLabel.height

                    Label {
                        id: fileSizeLabel

                        objectName: "sizeLabel"
                        font.pixelSize: Theme.fontSizeExtraSmall
                        color: fileDelegate.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                        text: Format.formatFileSize(model.size)
                    }

                    Label {
                        objectName: "fileTimeLabel"
                        anchors.right: parent.right
                        font.pixelSize: Theme.fontSizeExtraSmall
                        color: fileDelegate.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                        text: Format.formatDate(model.lastChanges, root.timeFormat)
                    }
                }
            }

            function deleteFile() {
                var currentRow = index;
                remorseDelete(function () {
                        filesModel.removeFile(model.path);
                    });
            }

            menu: Component {
                ContextMenu {
                    objectName: "contexMenu"

                    MenuItem {
                        objectName: "shareMenu"
                        text: qsTr("Share")
                        onClicked: {
                            shareAction.resources = [Qt.resolvedUrl(model.path)];
                            shareAction.trigger();
                        }
                    }
                    MenuItem {
                        objectName: "deleteMenu"
                        text: qsTr("Delete")
                        onClicked: fileDelegate.deleteFile()
                    }
                }
            }
        }
    }

    ShareAction {
        id: shareAction
        objectName: "shareAction"
    }

    Connections {
        onFileOpenRequested: chooseFile(path)

        target: dbusAdaptor
    }

    states: [
        State {
            name: "loading"

            PropertyChanges {
                target: documentsNotFoundLabel
                visible: false
            }
            PropertyChanges {
                target: documentListLoadIndicator
                visible: true
                running: true
            }
        },
        State {
            name: "work"

            PropertyChanges {
                target: documentsNotFoundLabel
                visible: false
            }
            PropertyChanges {
                target: documentListLoadIndicator
                visible: false
                running: false
            }
        },
        State {
            name: "documentsNotFound"

            PropertyChanges {
                target: documentsNotFoundLabel
                visible: true
            }
            PropertyChanges {
                target: documentListLoadIndicator
                visible: false
                running: false
            }
        }
    ]

    Component {
        id: filePicker

        FilePickerPage {
            title: qsTr("Select PDF file")
            nameFilters: [ '*.pdf' ]
            showSystemFiles: false

            anchors.topMargin: SafeZoneRect.insets.top

            onSelectedContentChanged: {
                chooseFile(FileUtils.absoluteFilePathFromUrl(selectedContent));
            }
        }
    }
}
