// SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Share 1.0
import Aurora.Controls 1.0
import ru.auroraos.amberpdf 1.0
import ru.auroraos.PdfViewer 1.0
import Sailfish.Pickers 1.0
import "../components"

Page {
    id: root

    property string pdfPath
    property int _maxSize: Math.max(Screen.width, Screen.height)
    property bool fullscreenMode: false
    property bool noteCreatingMode: false
    property bool isNotePointChosen: false
    property bool isDrawing: false
    property bool isDrawingPageChosen: false
    property bool isTextHighlighting: false

    readonly property var statesNames: ["correctWork", "loadingDocument", "badFile", "fileNotFound", "noteCreating", "drawing", "textHighlighting"]

    backNavigation: state !== statesNames[4] && state !== statesNames[5] && state !== statesNames[6]

    function checkCurrentFile() {
        fileInfo.refresh();
        if (fileInfo.isExist())
            return;
        state = root.statesNames[3];
    }

    function splitDocumentName() {
        return fileInfo.fileName + fileInfo.lastModified + fileInfo.size

    }

    function loadViewPosition() {
        var zoom = DocumentSettings.getDocumentZoom(splitDocumentName())
        var position = DocumentSettings.getDocumentPosition(splitDocumentName())

        pdfPagesView.zoom = zoom
        pdfPagesView.contentX = position.x
        pdfPagesView.contentY = position.y
        pdfPagesView.correctPosition()
    }

    onStatusChanged: {
        if (pageStack.currentPage.objectName === objectName) {
            if (status === PageStatus.Active && state === statesNames[1]) {
                pdfPagesView.pdfPath = pdfPath;
            }
        }
    }

    allowedOrientations: Orientation.All
    objectName: "contentPage"
    state: root.statesNames[1]

    onOrientationChanged: {
        if (pdfPagesView.isPagedView) {
            pdfPagesView.goToPage(pdfPagesView.currentIndex);
        }
        pdfPagesView.needGoToPage = true;
    }

    TextArea {
        id: errorText

        width: parent.width
        visible: root.state !== root.statesNames[0]
        color: Theme.highlightColor
        readOnly: true
        horizontalAlignment: TextEdit.AlignHCenter
        font.pixelSize: Theme.fontSizeExtraLarge
        anchors {
            verticalCenter: parent.verticalCenter
            horizontalCenter: parent.horizontalCenter
        }

        background: Rectangle {
            color: "transparent"
            border.color: "transparent"
        }
    }

    PdfView {
        id: pdfPagesView

        property real previousStatus: PdfDocument.Null

        documentProvider: pdfiumProvider
        clip: true
        width: parent.width
        catchBound: appBar.height
        annotationsPaint: true
        notesPaint: true
        anchors.horizontalCenter: parent.horizontalCenter
        deviceOrientation: root.orientation

        onClickedUrl: Qt.openUrlExternally(url)

        onNoteActivate: pageStack.push(Qt.resolvedUrl("NotePage.qml"), {
                                           noteText: noteText,
                                           author: author,
                                           pageNumber: pageIndex + 1,
                                           color: color,
                                           noteId: annotationId,
                                           pdfPagesView: pdfPagesView,
                                           noteType: noteType,
                                           editable: pdfPagesView.isPageInPainted(pageIndex)
                                       })

        onStatusChanged: {
            progressBar.value = status;
            if (previousStatus === status)
                return
            switch (previousStatus) {
            case PdfDocument.Null:
                root.state = (status === PdfDocument.Loading ? root.statesNames[1] : root.statesNames[2])
                break
            case PdfDocument.Loading:
                loadViewPosition()
                timerLoadViewPosition.start()

                timer.start()
                root.state = (status === PdfDocument.Ready ? root.statesNames[0] : root.statesNames[2])
                break
            case PdfDocument.Ready:
                root.state = root.statesNames[3]
                previousStatus = PdfDocument.Error
                break
            }

            if (previousStatus !== PdfDocument.Error)
                previousStatus = status
        }

        onHolding: {
            if (!root.noteCreatingMode && !root.isDrawing && !root.isTextHighlighting)
                return;
            if (isNotePointChosen)
                return
            if (isDrawingPageChosen)
                return
            if (root.isDrawing) {
                pageChoosing = false;
                var chosenPage = pdfPagesView.getPageByPoint(screenCoordinates);
                isDrawingPageChosen = true;
                pdfPagesView.startDrawing(chosenPage);
                return;
            }
            if (root.isTextHighlighting) {
                pageChoosing = false;
                var chosenPage = pdfPagesView.getPageByPoint(screenCoordinates);
                if (chosenPage)
                    pdfPagesView.goToPage(chosenPage);
                isDrawingPageChosen = true;
                pdfPagesView.startTextHighlighting(chosenPage);
                return;
            }
            if (root.noteCreatingMode) {
                pageChoosing = false;
                var chosenPage = pdfPagesView.getPageByPoint(screenCoordinates);
                if (chosenPage)
                    pdfPagesView.goToPage(chosenPage);
                isDrawingPageChosen = true;
                pdfPagesView.startNoteCreating(chosenPage);
                return;
            }
        }

        onCreateNote: {
            var createNotePage = pageStack.push(Qt.resolvedUrl("CreateNotePage.qml"), {isNewAnnotation: true});
            createNotePage.onCreateNote.connect(function (text, color, noteType) {
                pdfPagesView.addAnnotationAtPoint(screenCoordinates, "defaultuser", text, color, noteType, pdfPagesView.chosenPage);
                root.state = root.statesNames[0];
                pdfPagesView.stopNoteCreating();
            });
        }

        Binding {
            target: pdfPagesView
            property: "height"
            value: appBar.opened
                   ? root.height - appBar.height - toolBar.height
                   : root.height
        }
        Binding {
            target: pdfPagesView
            property: "y"
            value: appBar.opened
                   ? appBar.height
                   : 0
        }

        Behavior on y {
            enabled: !timer.running
            NumberAnimation {
                duration: root.state === root.statesNames[0] ? toolBar.animationDuration : 0
                easing.type: Easing.InOutQuad
            }
        }

        Timer {
            id: timer

            running: false
            repeat: false
            interval: 1000
        }

        Timer {
            id: timerLoadViewPosition

            onTriggered: loadViewPosition()

            interval: 150
        }
    }

    ContentHeader {
        id: appBar

        visibleButtons: root.state === root.statesNames[0]
        fullscreenMode: root.fullscreenMode
        noteCreatingMode: root.noteCreatingMode
        isDrawing: root.isDrawing && root.isDrawingPageChosen
        isTextHighlighting: root.isTextHighlighting && root.isDrawingPageChosen

        title: fileInfo.fileName

        onOpenDetails: pageStack.push(Qt.resolvedUrl("DetailsPage.qml"), {
                                          "pageCount":
                                            root.state === root.statesNames[1]
                                            ? qsTr("Loading")
                                            : pdfPagesView.count,
                                          "fileInfo": fileInfo
                                      })

        onFullscreenClicked: {
            root.fullscreenMode = true
        }
        onCancelNoteCreatingClicked: {
            pdfPagesView.stopNoteCreating();
            root.state = root.statesNames[0];
        }

        onDrawAccepted: {
            pdfPagesView.stopDrawing()
            pdfPagesView.drawInkAnnotation("defaultuser", "content")
            pdfPagesView.painter.reset()
            root.state = root.statesNames[0]
        }
        onDrawDeclined: {
            pdfPagesView.stopDrawing()
            pdfPagesView.painter.reset()
            root.state = root.statesNames[0]
        }
        onTextHighlightAccepted: {
            pdfPagesView.stopTextHighlight();
            root.state = root.statesNames[0];
        }
    }

    ToolBar {
        id: toolBar

        anchors.bottom: parent.bottom

        width: parent.width
        opened: !root.fullscreenMode
        pageMovingControlEnabled: pdfPagesView.isPagedView

        visible: (pdfPagesView.status === PdfDocument.Ready) && !root.isDrawing && !root.isTextHighlighting && !root.noteCreatingMode

        pageIndex: pdfPagesView.currentIndex
        pagesCount: pdfPagesView.count

        onOpenPopupMenu: popupMenu.openPopup(item)
        onOpenPopupNotes: popupMenuNotes.openPopup(item)

        onOpenNavigationPage: {
           var navigationPage = pageStack.push(Qt.resolvedUrl("NavigationPage.qml"), {
                                                    count: pdfPagesView.count,
                                                    sections: pdfPagesView.bookmarksModel,
                                                    bookmarks: pdfPagesView.userBookmarksModel
                                                })
           navigationPage.pageSelected.connect(function(page) { pdfPagesView.goToPage(page - 1) })
           navigationPage.sectionSelected.connect(function(page) {
               pdfPagesView.goToPage(page)
           })
           navigationPage.bookmarkSelected.connect(function(page) {pdfPagesView.goToPage(page - 1)})
           navigationPage.bookmarkDeleted.connect(function(page) {pdfPagesView.deleteUserBookmarkForPage(page)})
        }
        onNextSearchResultClicked: {
            pdfPagesView.highlightNextSearchResult();
            toolBar.searchResultString = pdfPagesView.getCurrentSearchResult();
            toolBar.searchResultCountString = pdfPagesView.getSearchResultCountString();
        }
        onBackSearchResultClicked: {
            pdfPagesView.highlightPreviousSearchResult();
            toolBar.searchResultString = pdfPagesView.getCurrentSearchResult();
            toolBar.searchResultCountString = pdfPagesView.getSearchResultCountString();
        }
        onCloseSearchPanelClicked: {
            pdfPagesView.clearSearchResults();
        }
        onBackPageClicked: {
            pdfPagesView.goToPreviousPage()
        }
        onNextPageClicked: {
            pdfPagesView.goToNextPage()
        }
    }

    Button {
        onClicked: {
            fullscreenMode = false
        }

        anchors {
            top: parent.top
            topMargin: Theme.paddingMedium
            right: parent.right
            rightMargin: Theme.paddingMedium
        }

        opacity: root.fullscreenMode ? 1.0 : 0.0
        visible: opacity
        backgroundColor: Theme.rgba(Theme.highlightBackgroundColor, Theme.opacityOverlay)
        icon.source: "image://theme/icon-splus-collapse"

        Behavior on opacity {
            FadeAnimation {}
        }
    }

    PdfDocument {
        id: pdfiumProvider

        objectName: "pdfDocument"
    }

    Item {
        id: progressItem

        anchors {
            left: parent.left
            right: parent.right
            bottom: parent.bottom
        }
        visible: pdfPagesView.status !== PdfDocument.Ready
        height: Theme.itemSizeMedium

        Column {
            anchors.fill: parent
            ProgressBar {
                id: progressBar

                anchors {
                    left: parent.left
                    right: parent.right
                }
                minimumValue: PdfDocument.Null
                maximumValue: PdfDocument.Ready
            }
            Text {
                anchors.horizontalCenter: parent.horizontalCenter
                color: Theme.highlightColor
                font.pixelSize: Theme.fontSizeLarge
                horizontalAlignment: Text.AlignHCenter
                text: pdfPagesView.status === PdfDocument.Loading ? qsTr("Document is loading") : ""
            }
        }
    }

    BackgroundItem {
        id: penControl

        anchors {
            left: parent.left
            right: parent.right
            bottom: parent.bottom
        }
        height: Theme.itemSizeMedium
        visible: root.isDrawingPageChosen

        IconButton {
            id: interactButton

            anchors {
                right: penControlRow.left
                verticalCenter: parent.verticalCenter
            }
            icon.sourceSize: Qt.size(Theme.iconSizeSmallPlus, Theme.iconSizeSmallPlus)
            icon.source: pdfPagesView.interactive ? "image://theme/icon-s-edit" : "image://theme/icon-s-secure"
            onClicked: pdfPagesView.interactive = !pdfPagesView.interactive
        }

        Row {
            id: penControlRow

            anchors.centerIn: parent
            spacing: Theme.paddingLarge
            visible: !root.noteCreatingMode

            Rectangle {
                id: penSample

                anchors.verticalCenter: parent.verticalCenter
                height: pdfPagesView.lineWidth
                width: height
                radius: height / 2
                color: pdfPagesView.lineColor
            }

            Text {
                text: qsTr("Pen settings")
                color: Theme.primaryColor
                font.pixelSize: Theme.fontSizeLarge
                anchors.verticalCenter: penSample.verticalCenter

                MouseArea {
                    onClicked: {
                        if (isDrawing) {
                            var drawPenSettingsDialog = pageStack.push(Qt.resolvedUrl("../pages/PenSettingsPage.qml"), {
                                                                           "color": pdfPagesView.lineColor,
                                                                           "penSize": pdfPagesView.lineWidth,
                                                                       })
                            drawPenSettingsDialog.chosen.connect(function () {
                                pdfPagesView.lineColor = drawPenSettingsDialog.color;
                                pdfPagesView.lineWidth = drawPenSettingsDialog.penSize;
                            });
                        } else if (isTextHighlighting) {
                            var textHighlightPenSettingsDialog = pageStack.push(Qt.resolvedUrl("../pages/PenSettingsPage.qml"), {
                                                                                    "color": pdfPagesView.textHighlightColor,
                                                                                    "showPenSizeSlider": false
                                                                                });
                            textHighlightPenSettingsDialog.penSize = textHighlightPenSettingsDialog.penMaximumSize;
                            textHighlightPenSettingsDialog.chosen.connect(function () {
                                pdfPagesView.textHighlightColor = textHighlightPenSettingsDialog.color;
                            });
                        }
                    }

                    anchors.fill: parent
                }
            }
        }

        IconButton {
            id: undoLineButton

            anchors {
                left: penControlRow.right
                verticalCenter: parent.verticalCenter
            }
            visible: pdfPagesView.painter.canUndo && !root.noteCreatingMode
            icon.sourceSize: Qt.size(Theme.iconSizeSmallPlus, Theme.iconSizeSmallPlus)
            icon.source: "image://theme/icon-m-rotate-left"
            onClicked: pdfPagesView.painter.undo()
        }

        IconButton {
            id: eraserButton

            onClicked: {
                if (pdfPagesView.painter.eraseMode) {
                    pdfPagesView.painter.pauseDrawing = false;
                    pdfPagesView.painter.eraseMode = false;
                } else {
                    pdfPagesView.painter.pauseDrawing = true;
                    pdfPagesView.painter.eraseMode = true;
                }
            }

            anchors {
                left: undoLineButton.right
                leftMargin: Theme.paddingSmall - Theme.paddingLarge
                verticalCenter: parent.verticalCenter
            }

            visible: !root.isTextHighlighting && !root.noteCreatingMode
            icon.sourceSize: Qt.size(Theme.iconSizeSmallPlus, Theme.iconSizeSmallPlus)
            icon.source: pdfPagesView.painter.eraseMode ? "image://theme/icon-m-edit" : "../assets/icon-splus-eraser.svg"
        }
    }

    PopupMenu {
        id: popupMenu

        property var delegate

        function openPopup(d) {
            d.selected = true
            delegate = d
            open(d)
        }

        onOpenedChanged: delegate.selected = opened

        preferredWidth: applicationWindow.preferredWidthPopup

        PopupMenuCheckableItem {
            id: horizontalOption

            onCheckedChanged: {
                if (checked) {
                    pdfPagesView.orientation = Qt.Horizontal;
                } else {
                    pdfPagesView.orientation = Qt.Vertical;
                }
                DocumentSettings.setVerticalScroll(splitDocumentName(), pdfPagesView.orientation === Qt.Vertical);
            }

            enabled: !pdfPagesView.isPagedView

            checked: false
            text: qsTr("Scroll pages horizontally")
        }

        PopupMenuCheckableItem {
            onCheckedChanged: {
                if (checked) {
                    pdfPagesView.isPagedView = true;
                    pdfPagesView.orientation = Qt.Horizontal;
                } else {
                    pdfPagesView.isPagedView = false;
                    pdfPagesView.orientation = horizontalOption.checked ? Qt.Horizontal : Qt.Vertical;
                }
                DocumentSettings.setVerticalScroll(splitDocumentName(), pdfPagesView.orientation === Qt.Vertical);
            }

            checked: pdfPagesView.isPagedView
            text: qsTr("Paged view")
        }

        PopupMenuDividerItem {}

        PopupMenuItem {
            onClicked: {
                shareAction.resources = [Qt.resolvedUrl(fileInfo.path)]
                shareAction.trigger()
            }

            icon.source: "image://theme/icon-splus-share"
            text: qsTr("Share")
        }

        PopupMenuItem {
            onClicked: {
                var searchPage = pageStack.push(Qt.resolvedUrl(
                                                    "SearchPhrasePage.qml"), {
                                                    "pdfPagesView": pdfPagesView
                                                })
                searchPage.searchResultSelected.connect(function (resultIndex) {
                    pdfPagesView.findSelectedSearchResult(resultIndex)
                    toolBar.searchResultString = pdfPagesView.getCurrentSearchResult();
                    toolBar.searchResultCountString = pdfPagesView.getSearchResultCountString();
                    toolBar.showSearchPanel = true;
                })
                searchPage.searchEnded.connect(function () {
                    pdfPagesView.findSelectedSearchResult(0)
                    toolBar.searchResultString = pdfPagesView.getCurrentSearchResult();
                    toolBar.searchResultCountString = pdfPagesView.getSearchResultCountString();
                    toolBar.showSearchPanel = true;
                })
            }

            icon.source: "image://theme/icon-m-search"
            text: qsTr("Search by word")
        }

        PopupMenuItem {
            onClicked: {
                pageStack.push(folderPickerDialog)
            }

            icon.source: "image://theme/icon-m-save"
            text: qsTr("Save document")
        }
    }

    Component {
        id: folderPickerDialog

        FolderPickerPage {
            title: qsTr("Save into")
            onSelectedPathChanged: pdfPagesView.saveDocumentAs(selectedPath + "/" + fileInfo.fileName)
        }
    }

    PopupMenuNotes {
        id: popupMenuNotes

        onShowNotes: pdfPagesView.notesPaint = show
        onNoteActivate: pageStack.push(Qt.resolvedUrl("NotePage.qml"), {
                                           noteText: noteText,
                                           author: author,
                                           pageNumber: pageIndex + 1,
                                           color: color,
                                           noteId: annotationId,
                                           pdfPagesView: pdfPagesView,
                                           noteType: noteType,
                                           editable: pdfPagesView.isPageInPainted(pageIndex)
                                       })

        onCreateNote: {
            root.state = root.statesNames[4]
        }

        onDrawNote: {
            root.state = root.statesNames[5]
        }

        onHighlightText: {
            root.state = root.statesNames[6];
        }

        pagesWithNotesModel: PageWithNotesSortModel {
            filterParameter: PageWithNotesSortModel.Count
        }

        isShowNotes: pdfPagesView.notesPaint
        preferredWidth: applicationWindow.preferredWidthPopup
        Component.onCompleted: {
            pagesWithNotesModel.sourceModel = pdfPagesView.pagesWithNotesModel
        }
    }


    FileInfo {
        id: fileInfo

        objectName: "fileInfo"
        path: root.pdfPath
    }

    ShareAction {
        id: shareAction
        objectName: "shareAction"
    }

    states: [
        State {
            name: root.statesNames[0]

            PropertyChanges {
                target: pdfPagesView
                visible: true
            }

            PropertyChanges {
                target: appBar
                title: fileInfo.fileName
            }

            PropertyChanges {
                target: root
                noteCreatingMode: false
            }

            PropertyChanges {
                target: root
                isDrawing: false
            }

            PropertyChanges {
                target: root
                isTextHighlighting: false
            }

            PropertyChanges {
                target: root
                isDrawingPageChosen: false
            }
        },
        State {
            name: root.statesNames[1]

            PropertyChanges {
                target: pdfPagesView
                visible: false
            }

            PropertyChanges {
                target: root
                noteCreatingMode: false
            }
        },
        State {
            name: root.statesNames[2]

            PropertyChanges {
                target: root
                pdfPath: ""
            }
            PropertyChanges {
                target: errorText
                text: qsTr("Could not open document")
            }
            PropertyChanges {
                target: pdfPagesView
                enabled: false
            }
            PropertyChanges {
                target: toolBar
                opened: false
            }
        },
        State {
            name: root.statesNames[3]

            PropertyChanges {
                target: root
                pdfPath: ""
            }
            PropertyChanges {
                target: errorText
                text: qsTr("File not found")
            }
            PropertyChanges {
                target: pdfPagesView
                enabled: false
            }
            PropertyChanges {
                target: toolBar
                opened: false
            }

            PropertyChanges {
                target: root
                noteCreatingMode: false
            }
        },
        State {
            name: root.statesNames[4]

            PropertyChanges {
                target: appBar
                title: root.isDrawingPageChosen ? qsTr("Hold a screen point to create a new node") : qsTr("Hold screen to choose page")
            }
            PropertyChanges {
                target: root
                noteCreatingMode: true
            }
            PropertyChanges {
                target: root
                isNotePointChosen: false
            }
            PropertyChanges {
                target: root
                isDrawingPageChosen: false
            }
            PropertyChanges {
                target: pdfPagesView
                visible: true
            }
            PropertyChanges {
                target: pdfPagesView
                pageChoosing: true
            }
        },
        State {
            name: root.statesNames[5]

            PropertyChanges {
                target: appBar
                title: root.isDrawingPageChosen ? qsTr("Draw annotation"): qsTr("Hold screen to choose page")
            }
            PropertyChanges {
                target: root
                isDrawing: true
            }
            PropertyChanges {
                target: root
                isDrawingPageChosen: false
            }
            PropertyChanges {
                target: pdfPagesView
                visible: true
            }
            PropertyChanges {
                target: pdfPagesView
                pageChoosing: true
            }
            PropertyChanges {
                target: penSample
                height: pdfPagesView.lineWidth
            }
            PropertyChanges {
                target: penSample
                color: pdfPagesView.lineColor
            }
        },
        State {
            name: root.statesNames[6]

            PropertyChanges {
                target: appBar
                title: root.isDrawingPageChosen ? qsTr("Highlight text") : qsTr("Hold screen to choose page")
            }
            PropertyChanges {
                target: root
                isTextHighlighting: true
            }
            PropertyChanges {
                target: root
                isDrawingPageChosen: false
            }
            PropertyChanges {
                target: pdfPagesView
                visible: true
            }
            PropertyChanges {
                target: pdfPagesView
                pageChoosing: true
            }
            PropertyChanges {
                target: penSample
                height: Theme.iconSizeSmall
            }
            PropertyChanges {
                target: penSample
                color: pdfPagesView.textHighlightColor
            }
        }
    ]
}
