// SPDX-FileCopyrightText: 2023 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

import QtQuick 2.0
import Sailfish.Silica 1.0
import Aurora.Controls 1.0

Item {
    id: root

    property bool opened: true
    property int pagesCount
    property int pageIndex
    property bool showSearchPanel: false
    property string searchResultString: ""
    property string searchResultCountString: ""
    property bool pageMovingControlEnabled: false

    readonly property int animationDuration: 300

    signal openPopupMenu(var item)
    signal openPopupNotes(var item)
    signal openNavigationPage
    signal nextSearchResultClicked()
    signal backSearchResultClicked()
    signal closeSearchPanelClicked()
    signal nextPageClicked()
    signal backPageClicked()

    height: showSearchPanel ? 2 * Theme.itemSizeMedium : Theme.itemSizeMedium
    opacity: opened && visible ? 1.0 : 0.0

    Behavior on opacity { FadeAnimation { duration: animationDuration } }

    Behavior on height {
        NumberAnimation {
            duration: animationDuration
            easing.type: Easing.InOutQuad
        }
    }

    Rectangle {
        anchors.top: root.top

        width: parent.width
        height: Theme._lineWidth
        opacity: Theme.opacityFaint
        color: palette.primaryColor
    }

    Item {
        y: opened ? 0 : parent.height
        width: parent.width
        height: Theme.itemSizeMedium

        Behavior on y {
            NumberAnimation {
                duration: animationDuration
                easing.type: Easing.InOutQuad
            }
        }

        ToolbarButton {
            id: notesButton

            onClicked: openPopupNotes(notesButton)

            anchors {
                left: parent.left
                leftMargin: Theme.paddingLarge
                verticalCenter: parent.verticalCenter
            }
            icon.source: "image://theme/icon-splus-note"
        }

        IconButton {
            id: previousPageButton

            anchors {
                left: notesButton.right
                leftMargin: Theme.paddingLarge
                verticalCenter: parent.verticalCenter
            }
            icon.source: "image://theme/icon-m-back"
            visible: pageMovingControlEnabled
            onClicked: {
                backPageClicked()
            }
        }

        Button {
            id: pageCounter

            onClicked: root.openNavigationPage()

            anchors.centerIn: parent
            height: Theme.itemSizeMedium
            width: (root.width - Theme.horizontalPageMargin * 2) / 3
            backgroundColor: "transparent"
            border.color: Theme.rgba(Theme.primaryColor, Theme.opacityFaint)

            Row {
                anchors.centerIn: parent

                Label {
                    text: qsTr("%1 | ").arg(Math.max(1, pageIndex + 1))
                    font.pixelSize: Theme.fontSizeSmall
                    font.family: Theme.fontFamily
                }
                Label {
                    text: qsTr("%1").arg(pagesCount)
                    font.pixelSize: Theme.fontSizeSmall
                    font.family: Theme.fontFamily
                    color: Theme.rgba(Theme.primaryColor, Theme.opacityHigh)
                }
            }
        }

        ToolbarButton {
            id: moreButton

            onClicked: root.openPopupMenu(moreButton)

            anchors {
                right: parent.right
                rightMargin: Theme.paddingLarge
                verticalCenter: parent.verticalCenter
            }
            icon.source: "image://theme/icon-splus-more"
        }
        IconButton {
            id: nextPageButton

            anchors {
                right: moreButton.left
                rightMargin: Theme.paddingLarge
                verticalCenter: parent.verticalCenter
            }
            visible: pageMovingControlEnabled
            icon.source: "image://theme/icon-m-forward"

            onClicked: {
                nextPageClicked()
            }
        }
    }
    Row {
        id: searchInfo

        anchors {
            left: parent.left
            right: parent.right
        }
        height: Theme.itemSizeMedium
        y: Theme.itemSizeMedium


        TextField {
            id: searchResultPreview

            anchors.verticalCenter: parent.verticalCenter
            width: parent.width - nextSearchButton.width - closeSearchButton.width - backSearchButton.width
            enabled: false
            text: searchResultString
            labelVisible: false

            rightItem: Label {
                id: countLabel

                text: searchResultCountString
            }
        }

        IconButton {
            id: backSearchButton

            anchors.verticalCenter: parent.verticalCenter
            icon.source: "image://theme/icon-m-back"

            onClicked: backSearchResultClicked()
        }
        IconButton {
            id: nextSearchButton

            anchors.verticalCenter: parent.verticalCenter
            icon.source: "image://theme/icon-m-forward"

            onClicked: nextSearchResultClicked()
        }
        IconButton {
            id: closeSearchButton

            anchors.verticalCenter: parent.verticalCenter
            icon.source: "image://theme/icon-m-cancel"

            onClicked: {
                showSearchPanel = false;
                closeSearchPanelClicked();
            }
        }
    }
}
