// SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include "interface/userbookmark.h"

#include "userbookmarksmodel.h"

UserBookmarksModel::UserBookmarksModel(QObject *parent) : QAbstractListModel(parent) {  }

/*!
 * \brief Overrided QAbstractListModel method.
 * \param parent Model index.
 * \return Row count in the model.
 */
int UserBookmarksModel::rowCount(const QModelIndex &parent) const
{
    return parent.isValid() ? 0 : m_data.size();
}

/*!
 * \brief Overrided QAbstractListModel method.
 * \param index Model index.
 * \param role Role in model.
 * \return Relevant role data.
 */
QVariant UserBookmarksModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid() || index.row() >= m_data.size())
        return {  };

    const auto *bookmark = m_data.at(index.row());
    if (bookmark == nullptr)
        return {  };

    switch (role) {
        case PageNumberRole: return QVariant::fromValue(bookmark->pageNumber);
        case PageTextRole: return QVariant::fromValue(bookmark->pageText);
    }

    return {  };
}

/*!
 * \brief Overrided QAbstractListModel method.
 * \return Roles List.
 */
QHash<int, QByteArray> UserBookmarksModel::roleNames() const
{
    return {
        { PageNumberRole, "pageNumber" },
        { PageTextRole, "pageText" }
    };
}

/*!
 * \brief Refreshes the model with new data.
 * \param newData New vector with bookmarks.
 */
void UserBookmarksModel::setNewData(const QVector<UserBookmark *> &newData)
{
    beginResetModel();
    m_data.clear();
    m_data = newData;
    endResetModel();
}

bool UserBookmarksModel::isPageBookmarked(int pageIndex)
{
    auto it = std::find_if(std::begin(m_data), std::end(m_data), [pageIndex](const UserBookmark* b) {
        return b->pageNumber == pageIndex + 1;
    });
    return m_data.end() != it;
}
