/****************************************************************************
**
** Copyright (C) 2022-2025 Open Mobile Platform LLC.
** Contact: https://community.omprussia.ru/open-source
**
** This file is part of the AmberPDF-QML-Plugin project.
**
** $QT_BEGIN_LICENSE:BSD$
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Open Mobile Platform LLC copyright holder nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QFutureWatcher>

#include <amberpdf/pdfpage.h>
#include <amberpdf/pdfannotation.h>

#include <baseannotation.h>
#include <QPoint>

#include "pdfpageitem.h"


BaseAnnotation::AnnotationType pdfAnnotationTypeToBaseAnnotationType(PdfAnnotation::AnnotationType type) {
    switch (type) {
    case PdfAnnotation::Link:
        return BaseAnnotation::AnnotationType::Link;
    case PdfAnnotation::Highlight:
        return BaseAnnotation::AnnotationType::HighLight;
    case PdfAnnotation::Text:
        return BaseAnnotation::AnnotationType::Text;
    case PdfAnnotation::Square:
        return BaseAnnotation::AnnotationType::Square;
    case PdfAnnotation::Freetext:
        return BaseAnnotation::AnnotationType::Freetext;
    case PdfAnnotation::Line:
        return BaseAnnotation::AnnotationType::Line;
    case PdfAnnotation::Circle:
        return BaseAnnotation::AnnotationType::Circle;
    case PdfAnnotation::Polygon:
        return BaseAnnotation::AnnotationType::Polygon;
    case PdfAnnotation::PolyLine:
        return BaseAnnotation::AnnotationType::Polyline;
    case PdfAnnotation::Underline:
        return BaseAnnotation::AnnotationType::Underline;
    case PdfAnnotation::Squiggly:
        return BaseAnnotation::AnnotationType::Squiggly;
    case PdfAnnotation::Strikeout:
        return BaseAnnotation::AnnotationType::Strikeout;
    case PdfAnnotation::Stamp:
        return BaseAnnotation::AnnotationType::Stamp;
    case PdfAnnotation::Caret:
        return BaseAnnotation::AnnotationType::Caret;
    case PdfAnnotation::Ink:
        return BaseAnnotation::AnnotationType::Ink;
    case PdfAnnotation::Popup:
        return BaseAnnotation::AnnotationType::Popup;
    case PdfAnnotation::Fileattachment:
        return BaseAnnotation::AnnotationType::Fileattachment;
    case PdfAnnotation::Sound:
        return BaseAnnotation::AnnotationType::Sound;
    case PdfAnnotation::Movie:
        return BaseAnnotation::AnnotationType::Movie;
    case PdfAnnotation::Widget:
        return BaseAnnotation::AnnotationType::Widget;
    case PdfAnnotation::Screen:
        return BaseAnnotation::AnnotationType::Screen;
    case PdfAnnotation::Printermark:
        return BaseAnnotation::AnnotationType::Printermark;
    case PdfAnnotation::Trapnet:
        return BaseAnnotation::AnnotationType::Trapnet;
    case PdfAnnotation::Watermark:
        return BaseAnnotation::AnnotationType::Watermark;
    case PdfAnnotation::Threed:
        return BaseAnnotation::AnnotationType::Threed;
    case PdfAnnotation::Richmedia:
        return BaseAnnotation::AnnotationType::Richmedia;
    case PdfAnnotation::Xfawidget:
        return BaseAnnotation::AnnotationType::Xfawidget;
    case PdfAnnotation::Redact:
        return BaseAnnotation::AnnotationType::Redact;
    default:
        return BaseAnnotation::AnnotationType::Unknown;
    }
}

PdfPageItem::PdfPageItem(QSharedPointer<PdfPage> amberPage, QObject *parent) : BasePage(parent),
    m_amberPage(amberPage)
{
}

PdfPageItem::~PdfPageItem()
{
    m_amberPage.clear();
    m_annotations.clear();
}

QList<QSharedPointer<BaseAnnotation>> PdfPageItem::annotations() const
{
    return m_annotations;
}

void PdfPageItem::loadAnnotations()
{
    auto annotations = m_amberPage->annotations();
    auto *watcher = new QFutureWatcher<QList<QSharedPointer<QObject>>>(this->parent());
    watcher->setFuture(annotations);
    connect(watcher, &QFutureWatcher<QList<QSharedPointer<QObject>>>::finished, this, [this, watcher]() {
        if (watcher == nullptr)
            return;

        if (watcher->isFinished() && !watcher->isCanceled()) {
            if (!m_annotations.isEmpty()) {
                m_annotations.clear();
            }

            auto annotationIndex = -1;
            for (const auto &annotation : watcher->result()) {
                ++annotationIndex;

                auto pdfAnnotation = qobject_cast<QSharedPointer<PdfAnnotation>>(annotation);
                if (pdfAnnotation == nullptr)
                    continue;

                QSharedPointer<BaseAnnotation> baseAnnotation;
                if (pdfAnnotation->type() == PdfAnnotation::Link) {
                    baseAnnotation = QSharedPointer<BaseAnnotation>(new BaseAnnotation());
                    baseAnnotation->type = pdfAnnotation->uri().isEmpty()
                            ? BaseAnnotation::AnnotationType::Link
                            : BaseAnnotation::AnnotationType::Url;

                    baseAnnotation->linkToPage = pdfAnnotation->linkToPage();
                    baseAnnotation->pageCoordinate = pdfAnnotation->linkPosition();
                    baseAnnotation->content = pdfAnnotation->uri();
                    baseAnnotation->isSearchResult = false;
                }
                if (pdfAnnotation->type() == PdfAnnotation::Highlight
                        || pdfAnnotation->type() == PdfAnnotation::Text
                        || pdfAnnotation->type() == PdfAnnotation::Square
                        || pdfAnnotation->type() == PdfAnnotation::Ink
                        || pdfAnnotation->type() == PdfAnnotation::PolyLine
                        || pdfAnnotation->type() == PdfAnnotation::Polygon) {
                    baseAnnotation = QSharedPointer<BaseAnnotation>(new BaseAnnotation());
                    baseAnnotation->type = pdfAnnotationTypeToBaseAnnotationType(pdfAnnotation->type());
                    baseAnnotation->lineWidth = pdfAnnotation->lineWidth();
                    if (pdfAnnotation->points().size() > 0)
                        baseAnnotation->pointLists = pdfAnnotation->points();
                    baseAnnotation->author =
                            pdfAnnotation->values()
                                    .value(pdfAnnotation->annotationKeyToQString(PdfAnnotation::T))
                                    .toString();
                    if (baseAnnotation->author == nullptr) {
                        continue;
                    } else {
                        baseAnnotation->isSearchResult =
                                baseAnnotation->author == PdfAnnotation::TEXT_SEARCH_AUTHOR_NAME;
                    }
                    baseAnnotation->content = pdfAnnotation->values()
                                                      .value(pdfAnnotation->annotationKeyToQString(
                                                              PdfAnnotation::Contents))
                                                      .toString();
                    static auto correctColor = [](const QColor &color) -> bool {
                        return (color != Qt::white && color != Qt::black);
                    };

                    auto annotationColor = pdfAnnotation->color();
                    annotationColor.setAlpha(255);
                    if (correctColor(annotationColor)) {
                        baseAnnotation->color = annotationColor;
                    } else {
                        annotationColor = pdfAnnotation->interiorColor();
                        annotationColor.setAlpha(255);

                        if (correctColor(annotationColor)) {
                            baseAnnotation->color = annotationColor;
                        } else {
                            foreach (auto colorPair, pdfAnnotation->objectsColors()) {
                                colorPair.first.setAlpha(255);
                                if (correctColor(colorPair.first)) {
                                    annotationColor = colorPair.first;
                                    break;
                                }

                                colorPair.second.setAlpha(255);
                                if (correctColor(colorPair.second)) {
                                    annotationColor = colorPair.second;
                                    break;
                                }
                            }
                        }
                    }

                    baseAnnotation->color = annotationColor;
                    baseAnnotation->color.setAlpha(255);
                    baseAnnotation->attachedPoints = pdfAnnotation->attachmentPoints();
                }

                if (baseAnnotation != nullptr) {
                    baseAnnotation->rect = pdfAnnotation->rect();
                    baseAnnotation->annotationId = annotationIndex;
                    m_annotations.append(baseAnnotation);
                }
            }
            emit annotationsLoaded();
        }

        watcher->deleteLater();
    });
}

bool PdfPageItem::isAnnotationsSupport() const
{
    return true;
}

int PdfPageItem::pageNumber() const
{
    return m_amberPage ? m_amberPage->pageNumber() : -1;
}

QString PdfPageItem::text()
{
    return m_amberPage->text();
}

QFuture<QSizeF> PdfPageItem::originalSize()
{
    return m_amberPage ? m_amberPage->originalSize() : QFuture<QSizeF>();
}

QFuture<QImage> PdfPageItem::bitmapFull(qreal pageScale, int renderFlags) const
{
    return m_amberPage ? m_amberPage->bitmapFull(pageScale, renderFlags) : QFuture<QImage>();
}

QFuture<QSharedPointer<QImage>> PdfPageItem::bitmapPart(qreal pageScaleX, qreal pageScaleY, int renderFlags, qreal zoom, const QPointF &bias) const
{
    return m_amberPage ? m_amberPage->bitmapPart(pageScaleX, pageScaleY, renderFlags, zoom, bias) : QFuture<QSharedPointer<QImage>>();
}

void PdfPageItem::addAnnotation(const QRectF &rect, const QColor &color, const QString &author, const QString &content, const QString &annotationType)
{
    if (!m_amberPage)
        return;

    auto *watcher = new QFutureWatcher<bool>();
    connect(watcher, &QFutureWatcher<bool>::finished, this, [this, watcher]() {
        if (watcher == nullptr)
            return;

        if (watcher->isFinished() && !watcher->isCanceled())
            emit annotationAdded(true);
        else
            emit annotationAdded(false);

        watcher->deleteLater();
    });

    auto future = m_amberPage->addAnnotation(rect, color, author, content, annotationType);
    watcher->setFuture(future);
}

void PdfPageItem::drawInkAnnotation(const QRectF &rect, const QColor &color, const float penSize, const QList<QList<QPointF>> points, const QString &author, const QString &content)
{
    if (!m_amberPage)
        return;

    auto *watcher = new QFutureWatcher<bool>();
    connect(watcher, &QFutureWatcher<bool>::finished, this, [this, watcher]() {
        if (watcher == nullptr)
            return;

        if (watcher->isFinished() && !watcher->isCanceled())
            emit annotationAdded(true);
        else
            emit annotationAdded(false);

        watcher->deleteLater();
    });
    auto future = m_amberPage->drawInkAnnotation(rect, color, penSize, points, author, content, 1);
    watcher->setFuture(future);
}

void PdfPageItem::removeAnnotation(int annotationId)
{
    if (!m_amberPage)
        return;

    auto *watcher = new QFutureWatcher<bool>();
    connect(watcher, &QFutureWatcher<bool>::finished, this, [this, watcher, annotationId]() {
        if (watcher == nullptr)
            return;

        if (watcher->isFinished() && !watcher->isCanceled())
            emit annotationDelete(annotationId, watcher->result());
        else
            emit annotationDelete(annotationId, false);
    });

    watcher->setFuture(m_amberPage->removeAnnotation(annotationId));
}

void PdfPageItem::removeSearchResultAnnotations()
{
    if (!m_amberPage)
        return;

    auto *watcher = new QFutureWatcher<bool>();
    connect(watcher, &QFutureWatcher<bool>::finished, this, [this, watcher]() {
        if (watcher == nullptr)
            return;
        if (watcher->isFinished() && !watcher->isCanceled())
            emit annotationDelete(-1, watcher->result());
        else
            emit annotationDelete(-1, false);
    });

    watcher->setFuture(m_amberPage->removeTextSearchResult());
}

/*!
 * \brief Updates selected annotation (note) by calling for old instance delete and
 * create a new one with refreshed parameters.
 * \param noteId Annotation (note) identifier.
 * \param newContent New annotation (note) description.
 * \param newColor New annotation (note) color.
 * \param annotationType Annotation (note) type (can't be refreshed).
 */
void PdfPageItem::editNote(int noteId, const QString &newContent, const QColor &newColor, const QString &annotationType)
{
    QSharedPointer<BaseAnnotation> annotationToEdit;
    for (const auto &annotation : m_annotations) {
        if (annotation == nullptr)
            continue;

        if (annotation->annotationId == noteId) {
            annotationToEdit = annotation;
            break;
        }
    }

    if (annotationToEdit == nullptr)
        return;

    auto *watcher = new QFutureWatcher<bool>();

    removePreviousAnnotInstance(watcher, noteId);
    addNewAnnotInstance(watcher, noteId, annotationToEdit, newContent, newColor, annotationType);
}

/*!
 * \brief Accessory function used to call to delete previous annotation (note) instance.
 * \param watcher QFutureWatcher.
 * \param noteId Annotation (note) identifier.
 */
void PdfPageItem::removePreviousAnnotInstance(QFutureWatcher<bool> *watcher, int noteId)
{
    connect(watcher, &QFutureWatcher<bool>::finished, this, [this, watcher, noteId]() {
        if (watcher == nullptr)
            return;

        if (watcher->isFinished() && !watcher->isCanceled())
            removeAnnotation(noteId);

        watcher->deleteLater();
    });
}

/*!
 * \brief Accessory function used to call to create new annotation (note) instance.
 * \param watcher QFutureWatcher.
 * \param noteId Annotation (note) identifier.
 * \param annotationToEdit BaseAnnotation instance.
 * \param newContent New annotation (note) description.
 * \param newColor New annotation (note) color.
 * \param annotationType Annotation (note) type (can't be changed).
 */
void PdfPageItem::addNewAnnotInstance(QFutureWatcher<bool> *watcher,
                                      int noteId,
                                      QSharedPointer<BaseAnnotation> annotationToEdit,
                                      const QString &newContent,
                                      const QColor &newColor,
                                      const QString &annotationType)
{
    if (annotationToEdit->type == BaseAnnotation::AnnotationType::Ink) {
        // Ink annotation type requires a bit different creation handling.
        auto future = m_amberPage->drawInkAnnotation(annotationToEdit->rect,
                                                     newColor,
                                                     annotationToEdit->lineWidth,
                                                     annotationToEdit->pointLists,
                                                     annotationToEdit->author,
                                                     newContent,
                                                     1);
        watcher->setFuture(future);
    } else {
        auto future = m_amberPage->addAnnotation(annotationToEdit->rect,
                                                 newColor,
                                                 annotationToEdit->author,
                                                 newContent,
                                                 annotationType,
                                                 annotationToEdit->attachedPoints);
        watcher->setFuture(future);
    }
    connect(this, &PdfPageItem::annotationDelete, this, [this, noteId](int annotationId, bool result) {
        if (noteId != annotationId)
            return;

        emit annotationEdited(noteId, result);
    });
}

void PdfPageItem::highlightText(QPoint startPoint, QPoint endPoint, QColor color)
{
    if (!m_amberPage)
        return;

    auto *watcher = new QFutureWatcher<bool>();
    connect(watcher, &QFutureWatcher<bool>::finished, this, [this, watcher]() {
        if (watcher == nullptr)
            return;
        if (watcher->isFinished() && !watcher->isCanceled())
            emit annotationAdded(true);
        else
            emit annotationAdded(false);

        watcher->deleteLater();
    });

    watcher->setFuture(m_amberPage->highlightText(startPoint, endPoint, color));
}

QList<QRectF> PdfPageItem::findTextRects(QPoint startPoint, QPoint endPoint)
{
    if (!m_amberPage)
        return {};

    return m_amberPage->findTextRects(startPoint, endPoint);
}
