// SPDX-FileCopyrightText: 2022-2024 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

import QtQuick 2.0

MouseArea {
    id: areaRoot

    objectName: "painterArea"

    readonly property bool painting: pressed

    property bool canUndo: false
    property string lineColor: "red"
    property real lineWidth: 2
    property bool pauseDrawing: true
    property real scale: 1
    property alias lines: canvas.lines
    property bool continuousLine
    property bool eraseMode: false
    property var deletedLines: []

    Component.onCompleted: {
        canvas.lines.pop();
    }

    onScaleChanged: {
        for (var i = 0; i < canvas.lines.length; i++) {
            for (var j = 0; j < canvas.lines[i].length; j++) {
                canvas.lines[i].points[j].x *= areaRoot.scale;
                canvas.lines[i].points[j].y *= areaRoot.scale;
            }
        }
        areaRoot.lineWidth = root.lineWidth * pdfView.itemScale;
        redraw();
    }

    onLineColorChanged: redraw();
    onLineWidthChanged: redraw();
    onEraseModeChanged: eraseMode ? canUndo = deletedLines.length > 0 : canUndo = canvas.lines.length > 0

    function scalePoints(coefWidth, coefHeight) {
        for (var i = 0; i < canvas.lines.length; i++) {
            for (var j = 0; j < canvas.lines[i].points.length; j++) {
                canvas.lines[i].points[j].x *= coefWidth;
                canvas.lines[i].points[j].y *= coefHeight;
            }
        }
        redraw();
    }

    function undo() {
        if (eraseMode) {
            if (deletedLines.length < 1)
                return;
            draw(deletedLines.pop());
            canUndo = deletedLines.length > 0;
            redraw();
        } else {
            if (canvas.lines.length < 1)
                return;
            canvas.lines.pop();
            canUndo = canvas.lines.length > 0;
            redraw();
        }
    }

    function draw(line) {
        canvas.lines.push(line);
        redraw();
    }

    function redraw() {
        canvas.fullRedraw = true;
        canvas.requestPaint();
    }

    function reset() {
        canvas.lines = [];
        deletedLines = [];
        eraseMode = false;
        pauseDrawing = false;
        canUndo = false;
        redraw();
    }

    function removeLine(x, y) {
        if (!canvas.lines.length)
            return;
        for (var i = 0; i < canvas.lines.length; i++) {
            if (isPointNearLine(lines[i], x, y)) {
                deletedLines.push(lines[i]);
                canUndo = true;
                canvas.lines.splice(i, 1);
                redraw();
                return;
            }
        }
    }

    function isPointNearLine(line, x, y) {
        var distances = [];
        var num;
        for (var i = 0; i < line.points.length; i++) {
            num = Math.sqrt(Math.pow(x - line.points[i].x, 2) + Math.pow(y - line.points[i].y, 2));
            if (!isNaN(num))
                distances.push(num);
        }
        var distance = Math.min.apply(null, distances);
        return distance < Math.max(20, lineWidth);
    }

    onPressed: {
        if (eraseMode) {
            removeLine(mouse.x, mouse.y);
            return;
        }
        continuousLine = true;
        canvas.lastPaintedPoint = 0;
        canvas.lines.push({color: areaRoot.lineColor, points: [Qt.point(mouse.x, mouse.y)]});
    }

    onPositionChanged: {
        if (eraseMode) {
            removeLine(mouse.x, mouse.y);
            return;
        }
        if (pauseDrawing)
            return;
        if (!pressed)
            return;
        if (continuousLine) {
            if (mouse.x < 0 || mouse.y < 0 || mouse.x > areaRoot.width || mouse.y > areaRoot.height) {
                continuousLine = false;
                return;
            }
            canvas.lines[canvas.lines.length - 1].points.push(Qt.point(mouse.x, mouse.y));
            canvas.requestPaint();
        }
    }

    onReleased: {
        if (pauseDrawing)
            return;
        if (canvas.lines[canvas.lines.length - 1].points.length < 2) {
            canvas.lines.pop();
            redraw();
        } else {
            canUndo = true;
        }
    }

    Canvas {
        id: canvas

        objectName: "canvas"
        anchors.fill: parent

        renderStrategy: Canvas.Threaded

        property var lines: [[]]
        property bool fullRedraw: false
        property int lastPaintedPoint: 0

        function drawLinePoints(line, startPoint) {
            startPoint = startPoint || 0;
            context.lineWidth = areaRoot.lineWidth * pdfView.itemScale;
            context.beginPath();
            var point = line.points[startPoint];
            context.moveTo(point.x, point.y);
            while (++startPoint < line.points.length) {
                point = line.points[startPoint];
                context.lineTo(point.x, point.y);
            }
            context.strokeStyle = line.color;
            context.stroke();
        }

        onPaint: {
            if (!getContext("2d"))
                return;
            context.lineCap = "round";
            context.lineJoin = "round";
            if (fullRedraw) {
                fullRedraw = false;
                context.clearRect(region.x, region.y, region.width, region.height);
                lines.forEach(function (line) {
                        drawLinePoints(line, 0);
                    });
            } else {
                var line = lines[lines.length - 1];
                if (!line)
                    return;
                drawLinePoints(line, lastPaintedPoint);
                lastPaintedPoint = line.points.length - 1;
            }
        }
    }
}
