﻿/****************************************************************************
**
** Copyright (C) 2021 - 2022 Open Mobile Platform LLC.
** Contact: https://community.omprussia.ru/open-source
**
** This file is part of the AmberPDF project.
**
** $QT_BEGIN_LICENSE:BSD$
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Open Mobile Platform LLC copyright holder nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <pdfium/fpdfview.h>

#include "pdftaskqueue.h"
#include "pdfdocumentholder.h"
#include "pdfpagerendertask.h"

/*!
 * \class PdfPageRenderFullTask
 * \brief The PdfPageRenderFullTask class is a task for rendering full image of the page.
 * \inmodule AmberPDF
 * \ingroup tasks
 */

/*!
 * Parametrized constructor.
 * \a pageScale is scale of page image.
 * \a renderFlags pdfium flags for rendering.
 * \a pageSize is size of the page.
 * \a page is QSharedPointer to page for rendering.
 * \a documentHolder is QSharedPointer to PdfDocumentHolder object with document.
 * \a interface is QFutureInterface for future tasks.
 */
PdfPageRenderFullTask::PdfPageRenderFullTask(float pageScale, int renderFlags,
                                     const QSizeF pageSize, const QSharedPointer<fpdf_page_t__> &page,
                                     const QSharedPointer<PdfDocumentHolder> &documentHolder,
                                     const QFutureInterface<QImage> &interface) :
    m_pageScale(pageScale),
    m_renderFlags(renderFlags),
    m_pageSize(pageSize),
    m_page(page),
    m_documentHolder(documentHolder),
    m_interface(interface)
{  }

/*!
 * Destructor.
 */
PdfPageRenderFullTask::~PdfPageRenderFullTask() = default;

/*!
 * Starts process of rendering full page image.
 */
void PdfPageRenderFullTask::run()
{
    QImage image = QImage(static_cast<int>(m_pageSize.width() * m_pageScale),
                          static_cast<int>(m_pageSize.height() * m_pageScale),
                          QImage::Format_ARGB32);

    if (image.isNull()) {
        m_interface.reportFinished(&image);
        return;
    }

    if (m_interface.isCanceled() || PdfTaskQueue::instance().blockedId().contains(m_documentHolder->id())) {
        m_interface.reportFinished(&image);
        return;
    }

    image.fill(Qt::white);

    FPDF_BITMAP bitmap = FPDFBitmap_CreateEx(image.width(), image.height(), FPDFBitmap_BGRA, image.bits(), image.bytesPerLine());
    if (bitmap == nullptr) {
        FPDFBitmap_Destroy(bitmap);
        m_interface.reportFinished(&image);
        return;
    }

    FPDF_RenderPageBitmap(bitmap, m_page.data(), 0, 0, image.width(), image.height(), 0, m_renderFlags);

    FPDFBitmap_Destroy(bitmap);

    m_interface.reportFinished(&image);
}

/*!
 * Returns document id.
 */
int PdfPageRenderFullTask::id() const
{
    return m_documentHolder->id();
}

/*!
 * Cancels task.
 */
void PdfPageRenderFullTask::cancel() {  }

/*!
 * \class PdfPageRenderPartTask
 * \brief The PdfPageRenderPartTask class is a task for rendering part of the page.
 * \inmodule AmberPDF
 * \ingroup tasks
 */

/*!
 * Destructor.
 */
PdfPageRenderPartTask::~PdfPageRenderPartTask() = default;

/*!
 * Parametrized constructor.
 * \a pageScaleX is horizontal scale of page image.
 * \a pageScaleY is vertical scale of page image.
 * \a renderFlags pdfium flags for rendering.
 * \a zoom is zoom amount.
 * \a bias is rendering bias.
 * \a pageSize is size of the page.
 * \a pageIndex is index of page in document.
 * \a documentHolder is QSharedPointer to PdfDocumentHolder object with document.
 * \a interface is QFutureInterface for future tasks.
 */
PdfPageRenderPartTask::PdfPageRenderPartTask(float pageScaleX, float pageScaleY,
                                             int renderFlags, float zoom,
                                             const QPointF &bias, const QSizeF pageSize,
                                             const int pageIndex,
                                             const QSharedPointer<PdfDocumentHolder> &documentHolder,
                                             const QFutureInterface<QSharedPointer<QImage>> &interface) :
    m_pageScaleX(pageScaleX),
    m_pageScaleY(pageScaleY),
    m_renderFlags(renderFlags),
    m_zoom(zoom),
    m_bias(bias),
    m_pageSize(pageSize),
    m_pageIndex(pageIndex),
    m_documentHolder(documentHolder),
    m_interface(interface)
{  }

/*!
 * Starts the process of rendering part of the page.
 */
void PdfPageRenderPartTask::run()
{
    auto loadedPage = QSharedPointer<fpdf_page_t__>(FPDF_LoadPage(m_documentHolder->document().data(), m_pageIndex), [](fpdf_page_t__ *){  });
    m_image.reset(new QImage(static_cast<int>(m_pageSize.width() * m_pageScaleX),
                             static_cast<int>(m_pageSize.height() * m_pageScaleY),
                             QImage::Format_ARGB32));

    if (m_image->isNull()) {
        m_interface.reportFinished(&m_image);
        return;
    }

    if (m_interface.isCanceled() || PdfTaskQueue::instance().blockedId().contains(m_documentHolder->id())) {
        m_interface.reportFinished(&m_image);
        return;
    }

    m_image->fill(Qt::white);

    FPDF_BITMAP bitmap = FPDFBitmap_CreateEx(m_image->width(), m_image->height(), FPDFBitmap_BGRA, m_image->bits(), m_image->bytesPerLine());
    if (bitmap == nullptr) {
        FPDFBitmap_Destroy(bitmap);
        m_interface.reportFinished(&m_image);
        return;
    }

    FS_MATRIX matrix { static_cast<float>(m_zoom), 0, 0, static_cast<float>(m_zoom), static_cast<float>(m_bias.x()), static_cast<float>(m_bias.y()) };
    FS_RECTF clipRectF { 0, 0, float(m_image->width()), float(m_image->height()) };
    FPDF_RenderPageBitmapWithMatrix(bitmap, loadedPage.data(), &matrix, &clipRectF, m_renderFlags);

    FPDFBitmap_Destroy(bitmap);
    FPDF_ClosePage(loadedPage.data());

    m_interface.reportFinished(&m_image);
}

/*!
 * Cancels task.
 */
void PdfPageRenderPartTask::cancel() {  }

/*!
 * Returns document id.
 */
int PdfPageRenderPartTask::id() const
{
    return m_documentHolder->id();
}
