/****************************************************************************
**
** Copyright (C) 2022 Open Mobile Platform LLC.
** Contact: https://community.omprussia.ru/open-source
**
** This file is part of the AmberPDF project.
**
** $QT_BEGIN_LICENSE:BSD$
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Open Mobile Platform LLC copyright holder nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <pdfium/fpdfview.h>
#include <pdfium/fpdf_annot.h>

#include "pdfdocumentholder.h"

#include "pdfpageaddannotationtask.h"

namespace
{
    constexpr auto annotBorderHRadius = 0.0f;
    constexpr auto annotBorderVRadius = 0.0f;
    constexpr auto annotBorderWidth = 0.0f;
}

/*!
 * \class PdfPageAddAnnotationTask
 * \brief The PdfPageAddAnnotationTask class is a task for adding highlight annotation on the page.
 * \inmodule AmberPDF
 * \ingroup tasks
 */

/*!
 * Parametrized constructor.
 * \a page is QSharedPointer to page for adding image.
 * \a newAnnotation is initialized object of NewAnnotation type with parameters of annotation.
 * \a interface is QFutureInterface for future tasks.
 * \a document is QSharedPointer to PdfDocumentHolder object with document.
 */
PdfPageAddAnnotationTask::PdfPageAddAnnotationTask(
        const QSharedPointer<fpdf_page_t__> &page, const NewAnnotation &newAnnotation,
        const QFutureInterface<bool> &interface, const QSharedPointer<PdfDocumentHolder> &document)
    : m_page(page),
      m_newAnnotation(newAnnotation),
      m_interface(interface),
      m_documentHolder(document)
{
}

/*!
 * Destructor.
 */
PdfPageAddAnnotationTask::~PdfPageAddAnnotationTask() = default;

/*!
 * Starts process of adding highlight annotation on page.
 */
void PdfPageAddAnnotationTask::run()
{
    if (!m_documentHolder)
        return;

    if (m_interface.isCanceled())
        return;

    auto result = false;

    auto annotation = FPDFPage_CreateAnnot(m_page.data(), stringTypeToPdfiumType(m_newAnnotation.type));

    if (annotation == nullptr) {
        m_interface.reportFinished(&result);
        return;
    }

    FS_RECTF annotationRect;
    annotationRect.bottom = m_newAnnotation.rect.bottom();
    annotationRect.top = m_newAnnotation.rect.top();
    annotationRect.left = m_newAnnotation.rect.left();
    annotationRect.right = m_newAnnotation.rect.right();
    result = FPDFAnnot_SetRect(annotation, &annotationRect);
    if (!result) {
        m_interface.reportFinished(&result);
        return;
    }

    if (stringTypeToPdfiumType(m_newAnnotation.type) == FPDF_ANNOT_HIGHLIGHT) {
        result = FPDFAnnot_SetColor(annotation,
                                    FPDFANNOT_COLORTYPE_Color,
                                    m_newAnnotation.color.red(),
                                    m_newAnnotation.color.green(),
                                    m_newAnnotation.color.blue(),
                                    m_newAnnotation.color.alpha());
    } else {
        result = FPDFAnnot_SetColor(annotation,
                                    FPDFANNOT_COLORTYPE_InteriorColor,
                                    m_newAnnotation.color.red(),
                                    m_newAnnotation.color.green(),
                                    m_newAnnotation.color.blue(),
                                    m_newAnnotation.color.alpha());
    }

    if (!result) {
        m_interface.reportFinished(&result);
        return;
    }

    result = FPDFAnnot_SetStringValue(annotation, "T", m_newAnnotation.author.utf16());
    if (!result) {
        m_interface.reportFinished(&result);
        return;
    }

    result = FPDFAnnot_SetStringValue(annotation, "Contents", m_newAnnotation.content.utf16());
    if (!result) {
        m_interface.reportFinished(&result);
        return;
    }

    result = FPDFAnnot_SetBorder(annotation, annotBorderHRadius, annotBorderVRadius, annotBorderWidth);
    if (!result) {
        m_interface.reportFinished(&result);
        return;
    }

    for (const auto attachedRect : m_newAnnotation.attachedPoints) {
        FS_QUADPOINTSF quad;
        quad.x1 = attachedRect.left();
        quad.y1 = attachedRect.top();
        quad.x2 = attachedRect.right();
        quad.y2 = attachedRect.top();
        quad.x3 = attachedRect.left();
        quad.y3 = attachedRect.bottom();
        quad.x4 = attachedRect.right();
        quad.y4 = attachedRect.bottom();
        result = FPDFAnnot_AppendAttachmentPoints(annotation, &quad);

        if (!result) {
            m_interface.reportFinished(&result);
            return;
        }
    }

    FPDFPage_CloseAnnot(annotation);

    m_interface.reportFinished(&result);
}

/*!
 * Cancels task.
 */
void PdfPageAddAnnotationTask::cancel() { }

/*!
 * Returns document id.
 */
int PdfPageAddAnnotationTask::id() const
{
    return m_documentHolder->id();
}

/*!
 * Converts \a type to one of pdfium annotation type.
 * Returns pdfium annotation type.
 */
int PdfPageAddAnnotationTask::stringTypeToPdfiumType(QString type)
{
    if (type == "Link")
        return FPDF_ANNOT_LINK;
    if (type == "Url")
        return FPDF_ANNOT_LINK;
    if (type == "HighLight")
        return FPDF_ANNOT_HIGHLIGHT;
    if (type == "Text")
        return FPDF_ANNOT_TEXT;
    if (type == "Square")
        return FPDF_ANNOT_SQUARE;
    if (type == "Unknown")
        return FPDF_ANNOT_UNKNOWN;
    return FPDF_ANNOT_UNKNOWN;
}
