// SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include <QStandardPaths>
#include <QFile>

#include "pdfdocumentholder.h"
#include "pdftaskqueue.h"
#include <pdfium/fpdf_doc.h>
#include <pdfium/fpdfview.h>

#include "pdfdocumentidloadtask.h"

/*!
 * \class PdfDocumentIdLoadTask
 * \brief The PdfDocumentIdLoadTask class is a task for getting document id.
 * \inmodule AmberPDF
 * \ingroup tasks
 */

/*!
 * Returns file id that was found in the active document (if was found).
 */
static QString getFileId(fpdf_document_t__ *document)
{
    std::vector<char> buffer(256);

    unsigned long bufLen = FPDF_GetFileIdentifier(document, FILEIDTYPE_CHANGING, buffer.data(), 256);

    if (bufLen == 0) {
        return "";
    }

    QByteArray byteArray(buffer.data(), bufLen);

    // Check for the null byte
    if (!byteArray.isEmpty() && byteArray[byteArray.size() - 1] == '\0') {
        byteArray.chop(1);
    }

    // Check if the byteArray has readable symbols.
    bool isReadable = true;
    for (char c : byteArray) {
        if (!isprint(static_cast<unsigned char>(c))) {
            isReadable = false;
            break;
        }
    }

    if (isReadable) {
        return QString::fromLatin1(byteArray);
    } else {
        return QString::fromLatin1(byteArray.toHex());
    }
}

/*!
 * Parametrized constructor.
 * \a documentHolder is QSharedPointer to PdfDocumentHolder object with document.
 * \a interface is QFutureInterface for future tasks.
 */
PdfDocumentIdLoadTask::PdfDocumentIdLoadTask(const QSharedPointer<PdfDocumentHolder> &documentHolder,
                                             const QFutureInterface<QString> &interface)
    : m_documentHolder(documentHolder), m_interface(interface)
{
}

/*!
 * Starts the process of getting file id.
 */
void PdfDocumentIdLoadTask::run()
{
    if (m_interface.isCanceled() || PdfTaskQueue::instance().blockedId().contains(m_documentHolder->id()))
        return;

    m_fileId = getFileId(m_documentHolder->document().data());
    m_interface.reportFinished(&m_fileId);
}

/*!
 * Cancels task.
 */
void PdfDocumentIdLoadTask::cancel() {}

/*!
 * Returns document id.
 */
int PdfDocumentIdLoadTask::id() const
{
    return m_documentHolder->id();
}
