// SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include <QStandardPaths>
#include <QFile>
#include <QJsonDocument>
#include <QJsonObject>

#include "pdfdocumentholder.h"
#include "pdftaskqueue.h"

#include "pdfdocumentdeleteuserbookmarktask.h"

/*!
 * Handles the process of removing user bookmark from json file.
 */
static void removeBookmark(int pageNumber, const QString &fileName)
{
    QString cachePath = QStandardPaths::writableLocation(QStandardPaths::CacheLocation);
    QString filePath = cachePath + "/" + fileName;

    QFile file(filePath);
    if (!file.open(QIODevice::ReadOnly)) {
        return;
    }

    QByteArray jsonData = file.readAll();
    file.close();

    QJsonDocument doc = QJsonDocument::fromJson(jsonData);
    if (!doc.isObject()) {
        return;
    }

    QJsonObject rootObj = doc.object();
    if (!rootObj.contains("bookmarks") || !rootObj["bookmarks"].isObject()) {
        return;
    }

    QJsonObject bookmarksObj = rootObj["bookmarks"].toObject();
    QString key = QString::number(pageNumber);

    if (!bookmarksObj.contains(key)) {
        return;
    }

    bookmarksObj.remove(key);
    rootObj["bookmarks"] = bookmarksObj;

    if (!file.open(QIODevice::WriteOnly)) {
        return;
    }

    file.write(QJsonDocument(rootObj).toJson(QJsonDocument::Indented));
    file.close();
}

/*!
 * \class PdfDocumentDeleteUserBookmarkTask
 * \brief The PdfDocumentDeleteUserBookmarkTask class is a task for removing user bookmarks.
 * \inmodule AmberPDF
 * \ingroup tasks
 */

/*!
 * Parametrized constructor.
 * \a pageNumber index of page in document.
 * \a fileName name of document.
 * \a interface is QFutureInterface for future tasks.
 * \a document is QSharedPointer to PdfDocumentHolder object with document.
 */
PdfDocumentDeleteUserBookmarkTask::PdfDocumentDeleteUserBookmarkTask(int pageNumber,
                                                             const QString &fileName,
                                                             const QFutureInterface<bool> &interface,
                                                             const QSharedPointer<PdfDocumentHolder> &document)
: m_interface(interface), m_documentHolder(document), m_pageNumber(pageNumber), m_fileName(fileName)
{
}

/*!
 * Starts the process of deleting user bookmark.
 */
void PdfDocumentDeleteUserBookmarkTask::run()
{
    if (m_interface.isCanceled() || PdfTaskQueue::instance().blockedId().contains(m_documentHolder->id()))
        return;

    removeBookmark(m_pageNumber, m_fileName);
    m_interface.reportFinished();
}

/*!
 * Cancels task.
 */
void PdfDocumentDeleteUserBookmarkTask::cancel() {}

/*!
 * Returns document id.
 */
int PdfDocumentDeleteUserBookmarkTask::id() const
{
    return m_documentHolder->id();
}
