/*******************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
** This file is part of the OfficeViewer project.
**
** Redistribution and use in source and binary forms,
** with or without modification, are permitted provided
** that the following conditions are met:
**
** * Redistributions of source code must retain the above copyright notice,
**   this list of conditions and the following disclaimer.
** * Redistributions in binary form must reproduce the above copyright notice,
**   this list of conditions and the following disclaimer
**   in the documentation and/or other materials provided with the distribution.
** * Neither the name of the copyright holder nor the names of its contributors
**   may be used to endorse or promote products derived from this software
**   without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
** THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
** FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
** EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*******************************************************************************/


import QtQuick 2.6
import Sailfish.Silica 1.0
import Nemo.FileManager 1.0
import "utils.js" as Utils

Dialog {
    id: root

    property string path
    property string baseName
    property string sourceMimeType
    property alias fileExtension: comboBox.value

    canAccept: !fileName.errorHighlight

    DialogHeader {
        id: dialogHeader

        //% "Save"
        acceptText: qsTrId("docviewer-he-save")
        //% "Cancel"
        cancelText: qsTrId("docviewer-he-cancel")
    }

    SilicaFlickable {
        anchors {
            top: dialogHeader.bottom
            left: parent.left
            right: parent.right
            bottom: parent.bottom
        }

        contentHeight: content.height

        Column {
            id: content

            width: parent.width
            spacing: Theme.paddingLarge

            InfoLabel {
                horizontalAlignment: Text.AlignLeft
                text: qsTrId("docviewer-ti-save_as")
            }

            TextField {
                id: fileName

                property bool fileExist
                readonly property var regExp: new RegExp("[\/*\?<>\|]+")

                function checkFileName() {
                    fileExist = FileEngine.exists("%1/%2%3".arg(path).arg(text).arg(comboBox.value))
                    errorHighlight = (regExp.test(text) || fileExist)
                }

                width: parent.width
                font.pixelSize: Theme.fontSizeMedium
                label: errorHighlight
                       ? (fileExist
                          //% "The same name already exists"
                          ? qsTrId("docviewer-la-name_already_exist")
                            //% "Invalid file name"
                          : qsTrId("docviewer-la-invalid_filename"))
                         //% "Title"
                       : qsTrId("docviewer-la-title_save_as")

                text: baseName
                placeholderText: qsTrId("docviewer-la-save_as")
                onFocusChanged: if (focus) selectAll()
                EnterKey.iconSource: "image://theme/icon-m-enter-accept"
                EnterKey.onClicked: accept()
                onTextChanged: {
                    checkFileName()
                    root.baseName = text
                }
                Component.onCompleted: {
                    text = baseName
                    checkFileName()
                }
            }

            ComboBox {
                id: comboBox
                property var types: {
                    switch(Utils.docviewerType(sourceMimeType)) {
                    case "calc":
                        return [".ods", ".xls", ".xlsx"]
                    case "impress":
                        return [".odp", ".ppt", ".pptx"]
                    case "writer":
                        return [".odt", ".doc", ".docx"]
                    }
                }

                //% "Format file"
                label: qsTrId("docviewer-la-format_file")

                menu: ContextMenu {
                    Repeater {
                        model: comboBox.types
                        MenuItem {
                            text: modelData
                        }
                    }
                }

                onValueChanged: fileName.checkFileName()
            }
        }
    }
}
