/*******************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
** This file is part of the OfficeViewer project.
**
** Redistribution and use in source and binary forms,
** with or without modification, are permitted provided
** that the following conditions are met:
**
** * Redistributions of source code must retain the above copyright notice,
**   this list of conditions and the following disclaimer.
** * Redistributions in binary form must reproduce the above copyright notice,
**   this list of conditions and the following disclaimer
**   in the documentation and/or other materials provided with the distribution.
** * Neither the name of the copyright holder nor the names of its contributors
**   may be used to endorse or promote products derived from this software
**   without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
** THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
** FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
** EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*******************************************************************************/


import QtQuick 2.2
import Sailfish.Silica 1.0
import Sailfish.Silica.Background 1.0
import Sailfish.Silica.private 1.0 as Private

SilicaControl {
    id: popUpMenu

    property var menuItem
    property var footer
    property alias active: menuLoader.active
    property bool activeBlock: true
    property bool isShowFull
    property int margin: Theme.paddingMedium
    property int verticalPosition: Qt.AlignVCenter
    property int horizontalPosition: Qt.AlignHCenter
    property bool headerItemVisible
    property int minHeight: Theme.paddingLarge * heightRatio
    property int widthRatio: 18
    readonly property int cornerRadius: 12
    readonly property int heightRatio: 28

    signal closed

    Private.AnimatedLoader {
        id: menuLoader

        width: popUpMenu.width
        height: popUpMenu.height

        animating: menuAnimation.running

        active: false
        source: menuComponent

        onAnimate: {
            if (item) {
                item.percentageClosed = 1
                menuAnimation.target = item
                menuAnimation.from = item.percentageClosed
                menuAnimation.to = 0
                menuAnimation.restart()
            } else if (replacedItem) {
                menuAnimation.target = replacedItem
                menuAnimation.from = 0
                menuAnimation.to = 1
                menuAnimation.restart()
            }
        }
    }

    NumberAnimation {
        id: menuAnimation

        running: false
        duration: 200
        easing.type: Easing.InOutQuad
        property: "percentageClosed"
    }

    Component {
        id: menuComponent

        Rectangle {
            id: menuItem

            property real percentageClosed
            readonly property real menuTop: headerItem.y - menuFlickable.contentY
            readonly property real topPadding: Math.max(0, footerLoader.y - headerItem.height - (Screen.sizeCategory > Screen.Medium
                    ? contentLoader.height
                    : Math.min(contentLoader.height, popUpMenu.minHeight)))

            width: popUpMenu.width
            height: popUpMenu.height

            color: activeBlock ? Theme.rgba("black", Theme.opacityLow * (1 - percentageClosed)) : "transparent"

            Flickable {
                id: menuFlickable

                property int topPosition: popUpMenu.margin
                                        - (menuItem.percentageClosed * (height + menuItem.menuTop + popUpMenu.margin))
                property int centerPosition: popUpMenu.height / 2
                                             - height / 2
                                             - popUpMenu.margin
                property int bottomPosition: popUpMenu.height
                                             - height
                                             - popUpMenu.margin
                                             + (menuItem.percentageClosed * (height - menuItem.menuTop + popUpMenu.margin))

                x: popUpMenu.horizontalPosition === Qt.AlignRight
                   ? popUpMenu.width - width - popUpMenu.margin
                   : popUpMenu.horizontalPosition === Qt.AlignLeft
                     ? popUpMenu.margin
                     : popUpMenu.width / 2 - width / 2
                y: popUpMenu.verticalPosition === Qt.AlignTop ? topPosition : popUpMenu.verticalPosition === Qt.AlignBottom ? bottomPosition : centerPosition

                width: Math.min(
                           popUpMenu.width - Theme.paddingLarge * 2,
                           Math.max(
                               Theme.paddingLarge * widthRatio,
                               footerLoader.item ? footerLoader.item.implicitWidth : 0))
                height: Math.min(
                            popUpMenu.height - (2 * popUpMenu.margin),
                            headerItem.height + contentLoader.height + footerLoader.height)

                contentHeight: menuItem.topPadding + headerItem.height + contentLoader.height + footerLoader.height

                interactive: popUpMenu.active   // Don't handle mouse events during fade out.

                flickableDirection: Flickable.VerticalFlick
                boundsBehavior: Flickable.DragOverBounds

                onHeightChanged: popUpMenu.isShowFull = height === contentHeight
                onContentHeightChanged: popUpMenu.isShowFull = height === contentHeight

                onDragEnded: {
                    if (popUpMenu.verticalPosition === Qt.AlignBottom && contentY < -popUpMenu.margin - Theme.paddingLarge) {
                        topMargin = -contentY
                        popUpMenu.closed()
                    }
                    if (popUpMenu.verticalPosition === Qt.AlignTop && contentY > -popUpMenu.margin - Theme.paddingLarge) {
                        topMargin = -contentY
                        popUpMenu.closed()
                    }
                }

                Item {
                    id: headerItem

                    y: menuItem.topPadding
                       + Math.min(0, menuFlickable.contentY)
                       - Math.min(0, menuFlickable.contentHeight - menuFlickable.height - menuFlickable.contentY)

                    width: menuFlickable.width
                    height: headerItemVisible ? Theme.paddingLarge : 0 // The element is left for ease of porting, but it is not displayed
                }

                Private.AnimatedLoader {
                    id: contentLoader

                    y: headerItem.y + headerItem.height

                    width: menuFlickable.width
                    height: item ? item.height : 0
                    source: popUpMenu.menuItem

                    onInitializeItem: {
                        item.width = Qt.binding(function() { return menuFlickable.width })
                    }
                }

                children: [
                    Rectangle {
                        id: background

                        y: Math.max(0, headerItem.y - menuFlickable.contentY)
                        z: -1
                        width: footerLoader.width
                        height: footerLoader.y - y

                        color: Qt.tint(
                                    popUpMenu.palette.colorScheme === Theme.LightOnDark ? "black" : "white",
                                    Theme.rgba(popUpMenu.palette.primaryColor, Theme.opacityFaint))
                    },
                    Item {
                        id: decoratorParent

                        y: background.y + headerItem.height
                        width: footerLoader.width
                        height: footerLoader.y - y

                        VerticalScrollDecorator {
                            _forcedParent: parent
                            flickable: menuFlickable

                            _sizeRatio: decoratorParent.height / (menuFlickable.contentHeight - contentLoader.y - footerLoader.height)
                            y: menuFlickable.contentHeight > menuFlickable.height + headerItem.y
                                    ? ((decoratorParent.height - height)
                                        * Math.max(0, menuFlickable.contentY - headerItem.y)
                                        / (menuFlickable.contentHeight - menuFlickable.height - headerItem.y))
                                    : 0
                        }
                    },
                    Rectangle {
                        y: Math.max(0, headerItem.y - menuFlickable.contentY)
                        width: headerItem.width
                        height: headerItem.height + Theme.paddingMedium

                        color: background.color

                        Rectangle {
                            x: (headerItem.width - width) / 2
                            y: (headerItem.height - height)

                            width: Theme.itemSizeLarge
                            height: Theme.paddingSmall

                            radius: height / 2

                            color: popUpMenu.palette.primaryColor

                            opacity: menuFlickable.contentY < headerItem.y ? 1 : Theme.opacityLow
                        }
                    },
                    MouseArea {
                        anchors.fill: footerLoader
                    },
                    Private.AnimatedLoader {
                        id: footerLoader

                        y: menuFlickable.height - height

                        width: menuFlickable.width
                        height: item ? item.height: 0

                        source: popUpMenu.footer

                        onInitializeItem: {
                            item.width = Qt.binding(function() { return menuFlickable.width })
                        }
                    }
                ]
            }

            // To round the corners the menu is being rendered to a ShaderEffectSource which is
            // used as the source item for a Background item which composites the menu with the
            // rest of the UI with rounded corners. Using the ShaderEffectSource also means
            // overlapping items in the menu won't blend together when the menu fades in and out.
            ShaderEffectSource {
                id: menuShaderSource

                width: menuFlickable.width
                height: menuFlickable.height

                sourceItem: menuFlickable
                hideSource: true
                visible: false
            }

            Background {
                id: menuShaderItem

                // The ShaderEffectSourceItem has its size fixed to the maximum open size of the
                // menu, this is so each frame of animation doesn't have to allocate a new texture
                // of a different size when the menu expands. This matrix transforms the normalized
                // item rectangle coordinates to the visible sub rectangle of the source item.
                readonly property matrix4x4 menuSourceMatrix: Qt.matrix4x4(
                        1, 0, 0, 0,
                        0, height / menuFlickable.height, 0, 0,
                        0, 0, 1, 0,
                        0, menuItem.menuTop / menuFlickable.height, 0, 1)

                x: menuFlickable.x
                y: menuFlickable.y + menuItem.menuTop - (popUpMenu.verticalPosition === Qt.AlignTop
                                                         ? Math.max(0, menuFlickable.contentY)
                                                         : popUpMenu.verticalPosition === Qt.AlignBottom ? Math.min(0, menuFlickable.contentY) : 0)
                width: menuFlickable.width
                height: menuFlickable.height - menuItem.menuTop

                radius: popUpMenu.cornerRadius
                sourceItem: menuShaderSource
                fillMode: Background.Stretch

                material: Material {
                    vertexShader: "
                        attribute highp vec4 position;
                        attribute highp vec2 normalizedPosition;

                        uniform highp mat4 positionMatrix;

                        uniform highp mat4 menuSourceMatrix;

                        varying highp vec2 sourceCoord;

                        void backgroundMain() {
                            gl_Position = positionMatrix * position;
                            sourceCoord = (menuSourceMatrix * vec4(normalizedPosition, 0, 1)).xy;
                        }"
                    fragmentShader: "
                        uniform lowp sampler2D sourceTexture;

                        varying highp vec2 sourceCoord;

                        void backgroundMain() {
                            gl_FragColor = texture2D(sourceTexture, sourceCoord);
                        }"
                }
            }

            InverseMouseArea {
                anchors.fill: menuShaderItem
                enabled: popUpMenu.active && popUpMenu.activeBlock
                stealPress: true
                onPressedOutside: closed()
            }
        }
    }
}
