/*******************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
** This file is part of the OfficeViewer project.
**
** Redistribution and use in source and binary forms,
** with or without modification, are permitted provided
** that the following conditions are met:
**
** * Redistributions of source code must retain the above copyright notice,
**   this list of conditions and the following disclaimer.
** * Redistributions in binary form must reproduce the above copyright notice,
**   this list of conditions and the following disclaimer
**   in the documentation and/or other materials provided with the distribution.
** * Neither the name of the copyright holder nor the names of its contributors
**   may be used to endorse or promote products derived from this software
**   without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
** THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
** FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
** EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*******************************************************************************/



import QtQuick 2.6
import Sailfish.Silica 1.0

Row {
    id: root

    property bool selectAllEnabled
    property bool portrait: true

    property real _buttonWidth: width / (repeater.count + 1)

    readonly property var defaultButtons: {
        var buttons = []

        buttons.push({
            "icon": "image://theme/icon-splus-share",
            //% "Share"
            "label": qsTrId("docviewer-la-share"),
            "action": function () {
                clear()
                share()
            }
        })

        buttons.push({
            "icon": "image://theme/icon-splus-clipboard",
            //% "Copy"
            "label": qsTrId("docviewer-la-copy"),
            "action": function () {
                clear()
                copy()
            }
        })

        buttons.push({
            "icon": "image://theme/icon-m-input-remove",
            //% "Clear"
            "label": qsTrId("docviewer-la-clear"),
            "action": clear
        })

        return buttons
    }

    property var buttons: defaultButtons

    property int fontSize: {
        var size = Theme.fontSizeTiny

        if (!portrait && size > Theme.fontSizeTinyBase) {
            var string = ""
            for (var i = 0; i < buttons.length; ++i) {
                string = string + (buttons[i].label || "")
            }

            var requiredWidth = metrics.advanceWidth(string) + (buttons.length * (Theme.itemSizeSmall + (2 * horizontalSpacing)))

            if (requiredWidth >= width - leftPadding - rightPadding) {
                size = Theme.fontSizeTinyBase
            }
        }
        return size
    }
    property real horizontalSpacing

    signal share
    signal copy
    signal clear

    height: Theme.itemSizeSmall + Theme.paddingLarge

    FontMetrics {
        id: metrics
        font.pixelSize: Theme.fontSizeTiny
    }

    Repeater {
        id: repeater

        model: root.buttons
        delegate: root.portrait ? portraitButtonDelegate : landscapeButtonDelegate
    }

    Component {
        id: portraitButtonDelegate

        IconButton {
            id: portraitButton

            width: (root.width - root.leftPadding - root.rightPadding) / Math.max(1, repeater.count)
            height: root.height
            icon.source: modelData.icon
            icon.sourceSize: Qt.size(Theme.iconSizeSmallPlus, Theme.iconSizeSmallPlus)
            icon.anchors.verticalCenterOffset: -(portraitLabel.baselineOffset - Theme.paddingSmall)
            onClicked: root.buttons[index].action()

            Label {
                id: portraitLabel

                x: (portraitButton.width - width) / 2
                y: portraitButton.icon.y + portraitButton.icon.height - Theme.paddingSmall

                text: modelData.label
                font.pixelSize: root.fontSize
            }
        }
    }
    Component {
        id: landscapeButtonDelegate

        IconButton {
            id: landscapeButton

            width: (root.width - root.leftPadding - root.rightPadding) / Math.max(1, repeater.count)
            height: root.height
            icon.source: modelData.icon
            icon.anchors.horizontalCenterOffset: -(Math.min(landscapeLabel.width, landscapeLabel.implicitWidth) + root.horizontalSpacing) / 2

            onClicked: root.buttons[index].action()

            Label {
                id: landscapeLabel
                x: landscapeButton.icon.x + landscapeButton.icon.width + root.horizontalSpacing
                y: (landscapeButton.height - height) / 2
                width: landscapeButton.width - landscapeButton.icon.width - (2 * root.horizontalSpacing)

                text: modelData.label
                font.pixelSize: root.fontSize
                truncationMode: TruncationMode.Fade
            }
        }
    }
}
