/*******************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
** This file is part of the OfficeViewer project.
**
** Redistribution and use in source and binary forms,
** with or without modification, are permitted provided
** that the following conditions are met:
**
** * Redistributions of source code must retain the above copyright notice,
**   this list of conditions and the following disclaimer.
** * Redistributions in binary form must reproduce the above copyright notice,
**   this list of conditions and the following disclaimer
**   in the documentation and/or other materials provided with the distribution.
** * Neither the name of the copyright holder nor the names of its contributors
**   may be used to endorse or promote products derived from this software
**   without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
** THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
** FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
** EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*******************************************************************************/


import QtQuick 2.1
import Sailfish.Silica 1.0
import ".."

Item {
    id: handle

    property real fixedX
    property real fixedY
    property real scale
    property real minimumX
    property real maximumX

    property QtObject contentItem

    property bool moving
    readonly property bool dragActive: mouseArea.drag.active

    property string markerTag
    // Mask area for touch handling, should cover whole mousearea
    property rect mask: Qt.rect(0, 0, 0, 0)

    property bool _startMarker: markerTag === "start"

    width: Theme.itemSizeExtraSmall
    height: width

    onFixedXChanged: {
        if (!moving)
            timer.restart()

        _startMarker ? handle.x = fixedX - width : handle.x = fixedX
    }
    onFixedYChanged: {
        if (!moving)
            timer.restart()

        handle.y = fixedY
    }

    onDragActiveChanged: {
        if (dragActive) {
            moving = true
        } else {
            moving = false
            timer.restart()
        }
    }

    onVisibleChanged: {
        if (visible) {
            showAnimation.restart()
        } else {
            mask = Qt.rect(0, 0, 0, 0)
        }
    }

    Image {
        x: _startMarker ? handle.width - width : 0
        width: Theme.iconSizeSmallPlus
        height: width
        transformOrigin: _startMarker ? Item.TopRight : Item.TopLeft
        source: "image://theme/icon-m-textselection-" + markerTag
    }

    MouseArea {
        id: mouseArea

        property real previousX: -1
        property real previousY: -1

        anchors {
            fill: parent
            leftMargin: -Theme.paddingLarge
            rightMargin: -Theme.paddingLarge
            bottomMargin: -Theme.paddingLarge
        }

        drag {
            target: parent
            axis: Drag.XandYAxis
            minimumX: handle.minimumX - width + Theme.iconSizeSmallPlus
            minimumY: 0
            maximumX: handle.maximumX
            maximumY: contentItem.contentHeight
        }

        preventStealing: true

        onPositionChanged: {
            var targetX = mouse.x
            var targetY = mouse.y

            if (!handle.moving || Math.abs(targetX - previousX) < 0.99 && Math.abs(targetY - previousY) < 0.99) {
                return
            }

            if (_startMarker)
                contentItem.startSelectText(Qt.point(handle.x + handle.width, handle.y))
            else
                contentItem.endSelectText(Qt.point(handle.x, handle.y))

            previousX = targetX
            previousY = targetY
        }
    }

    NumberAnimation {
        id: showAnimation

        target: handle
        property: "opacity"
        from: 0
        to: 1.0
    }

    ParallelAnimation {
        id: targetPositionAnimation

        NumberAnimation {
            target: handle
            property: "x"
            to: _startMarker ? fixedX - width : fixedX
            duration: 100
            easing.type: Easing.InOutQuad
        }
        NumberAnimation {
            target: handle
            property: "y"
            to: handle.fixedY
            duration: 100
            easing.type: Easing.InOutQuad
        }
    }

    Timer {
        id: timer

        onTriggered: targetPositionAnimation.start()
        interval: 300
    }
}
