/*******************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
** This file is part of the OfficeViewer project.
**
** Redistribution and use in source and binary forms,
** with or without modification, are permitted provided
** that the following conditions are met:
**
** * Redistributions of source code must retain the above copyright notice,
**   this list of conditions and the following disclaimer.
** * Redistributions in binary form must reproduce the above copyright notice,
**   this list of conditions and the following disclaimer
**   in the documentation and/or other materials provided with the distribution.
** * Neither the name of the copyright holder nor the names of its contributors
**   may be used to endorse or promote products derived from this software
**   without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
** THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
** FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
** EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*******************************************************************************/


import QtQuick 2.0
import Sailfish.Silica 1.0
import Aurora.Office.Files 1.0
import ru.omp.documentprovider.libreoffice 1.0

Item {
    id: root

    Label {
        id: documentName

        anchors {
            left: parent.left
            right: parent.right
        }

        truncationMode: TruncationMode.Elide
        elide: Label.ElideMiddle
        font.pixelSize: Theme.fontSizeTiny
        color: Theme.primaryColor
        text: {
            if (!coverPrivate.docView) {
                return ""
            }

            return fileListModel.getFileInfo(coverPrivate.docView.path).fileName
        }
    }

    Loader {
        id: documentLoader

        anchors {
            fill: parent
            leftMargin: -(Theme.paddingMedium + coverPrivate.borderPadding)
            rightMargin: -(Theme.paddingMedium + coverPrivate.borderPadding)
            topMargin: documentName.height + Theme.paddingSmall
        }

        active: coverPrivate.docView

        sourceComponent: Item {
            id: previewRoot

            anchors.fill: parent

            clip: true

            Connections {
                target: __cover

                onStatusChanged: {
                    if (__cover.status === Cover.Activating) {
                        coverPrivate.grabDocViewAtChanges()
                    }
                }
            }

            Image {
                id: preview

                readonly property bool includeDocHeader: {
                    if (!coverPrivate.docView) {
                        return false
                    }

                    return coverPrivate.docView.contentY <= coverPrivate.docView.topMargin
                }

                readonly property real scaledDocTopMargin: {
                    if (!coverPrivate.docView) {
                        return 0
                    }

                    const scale = previewRoot.height / (coverPrivate.docView.height + coverPrivate.docView.topMargin)
                    const value = coverPrivate.docView.topMargin - coverPrivate.docView.contentY

                    return Math.round(value * scale) * 2
                }

                anchors {
                    fill: parent
                    topMargin: preview.includeDocHeader
                               ? -scaledDocTopMargin
                               : 0
                }

                asynchronous: true

                source: coverPrivate.grabResult ? coverPrivate.grabResult.url : ""

                fillMode: Image.PreserveAspectCrop

                verticalAlignment: Image.AlignTop
                horizontalAlignment: Image.AlignLeft

                opacity: (coverPrivate.grabbing && coverPrivate.positionChanged) ? 0 : 1

                Behavior on opacity {
                    FadeAnimation {}
                }
            }

            BusyIndicator {
                id: busyIndicator

                anchors.centerIn: parent
                visible: {
                    if (!coverPrivate.docView) {
                        return false
                    }

                    if (coverPrivate.docView.documentProvider.status === LoDocument.Loading) {
                        return true
                    }

                    return coverPrivate.grabbing && coverPrivate.positionChanged
                }

                running: visible
                _forceAnimation: true
            }

            HighlightImage {
                id: warningIcon

                anchors.centerIn: parent

                visible: {
                    if (!coverPrivate.docView) {
                        return false
                    }

                    if (busyIndicator.running) {
                        return false
                    }

                    return coverPrivate.docView.documentProvider.status === LoDocument.Error
                }

                source: "image://theme/icon-l-attention"
                highlighted: true
                anchors.horizontalCenter: parent.horizontalCenter
            }
        }
    }

    QtObject {
        id: coverPrivate

        readonly property int borderPadding: Theme.lineWidth
                                             ? Theme.lineWidth
                                             : 4

        readonly property var docView: applicationWindow.docView

        readonly property bool positionChanged: {
            if (!coverPrivate.docView) {
                return false
            }

            return coverPrivate.lastContentX !== coverPrivate.docView.contentX ||
                    coverPrivate.lastContentY !== coverPrivate.docView.contentY
        }

        property var grabResult: null

        property bool grabbing: false

        property int lastContentX: -1
        property int lastContentY: -1

        readonly property Timer delayedResultAssignTimer: Timer {
            property var _result: null

            function assign(result) {
                _result = result
                start()
            }

            function finish(result) {
                coverPrivate.grabResult = result

                coverPrivate.grabbing = false

                coverPrivate.updateLastContentPos()
            }

            interval: 200

            onTriggered: {
                finish(_result)
            }
        }

        function updateLastContentPos() {
            coverPrivate.lastContentX = coverPrivate.docView.contentX
            coverPrivate.lastContentY = coverPrivate.docView.contentY
        }

        function grabDocViewAtChanges() {
            if (!coverPrivate.docView ||
                    coverPrivate.docView.documentProvider.status !== LoDocument.Ready) {
                return
            }

            coverPrivate.grabDocView()
        }

        function grabDocView() {
            if (!applicationWindow.visible) {
                return
            }

            if (!coverPrivate.docView) {
                console.warn('null document view')
                return
            }

            coverPrivate.grabbing = true

            coverPrivate.docView.cancelFlick()

            coverPrivate.docView.grabToImage(function(result) {
                if (!coverPrivate.positionChanged) {
                    delayedResultAssignTimer.finish(result)
                } else {
                    coverPrivate.delayedResultAssignTimer.assign(result)
                }
            })
        }
    }
}
