/*******************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
** This file is part of the OfficeViewer project.
**
** Redistribution and use in source and binary forms,
** with or without modification, are permitted provided
** that the following conditions are met:
**
** * Redistributions of source code must retain the above copyright notice,
**   this list of conditions and the following disclaimer.
** * Redistributions in binary form must reproduce the above copyright notice,
**   this list of conditions and the following disclaimer
**   in the documentation and/or other materials provided with the distribution.
** * Neither the name of the copyright holder nor the names of its contributors
**   may be used to endorse or promote products derived from this software
**   without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
** THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
** FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
** EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*******************************************************************************/



#pragma once

#include <QQuickItem>

#include "docviewtypes.h"

class BasePage;
class BaseAnnotation;
class DocumentMapper;
class QSGTexture;
class MultiPageTile;
class BaseDocItem;
class DocItemAnnotation;
class DocItemComment;
class DocItemSearchHighlight;
class BackgroundPage;
class PageContainer : public QQuickItem
{
    Q_OBJECT

    Q_PROPERTY(QSizeF requestedSize READ requestedSize WRITE setRequestedSize NOTIFY requestedSizeChanged)
    Q_PROPERTY(Qt::Orientation orientation READ orientation WRITE setOrientation NOTIFY orientationChanged)
    Q_PROPERTY(qreal scale READ scale WRITE setScale NOTIFY scaleChanged)
    Q_PROPERTY(QColor highLightColor READ highLightColor WRITE setHighLightColor NOTIFY highLightColorChanged)
    Q_PROPERTY(bool annotationsPaint READ annotationsPaint WRITE setAnnotationsPaint NOTIFY annotationsPaintChanged)
    Q_PROPERTY(bool notesPaint READ notesPaint WRITE setNotesPaint NOTIFY notesPaintChanged)

public:
    explicit PageContainer(QQuickItem *parent = nullptr);
    ~PageContainer() override;

    QSGNode *updatePaintNode(QSGNode *oldNode, QQuickItem::UpdatePaintNodeData *) override;
    void geometryChanged(const QRectF &newGeometry, const QRectF &oldGeometry) override;

    QSizeF requestedSize() const;
    Qt::Orientation orientation() const;
    qreal scale() const;
    void setPageSource(QSharedPointer<BasePage> pageSource);
    void setPageGeometry(const PageGeometry &pg);
    int index() const;
    void setMapper(DocumentMapper *mapper);
    static QSizeF pageCurrentSize(const PageGeometry &pageGeometry, const QSizeF &requestedSize, Qt::Orientation orientation, qreal scale);
    void setVisibleArea(const QRectF &visibleArea);
    bool annotationsPaint() const;
    bool notesPaint() const;
    QSharedPointer<BasePage> source() const;
    void addAnnotation(const QRectF &rect, const QColor &color,
                       const QString &author, const QString &content);
    qreal imageScale() const;

    const QColor &highLightColor() const;
    void setHighLightColor(const QColor &newHighLightColor);
    qreal pageRate() const;

public slots:
    void setRequestedSize(QSizeF requestedSize);
    void setOrientation(Qt::Orientation orientation);
    void setScale(qreal scale);
    void setAnnotationsPaint(bool annotationsPaint);
    void setNotesPaint(bool notesPaint);
    void forceUpdateRect(const QRectF &visibleArea);
    void setColorSelected(const QColor &color);

signals:
    void requestedSizeChanged(QSizeF requestedSize);
    void orientationChanged(Qt::Orientation orientation);
    void scaleChanged(qreal scale);
    void pageReady();
    void pageChanged();
    void annotationsPaintChanged(bool annotationsPaint);
    void annotationsLoaded();
    void annotationActivate(BaseAnnotation *);
    void notesPaintChanged(bool notesPaint);
    void noteActivate(BaseAnnotation *);
    void cellActivate(BaseAnnotation *);
    void highLightColorChanged();

private slots:
    void _correctSize();
    void _tailorise();
    void _updateVisible();
    void _loadAnnotations();
    void _prepareBackgroundPage();

private:
    PageGeometry m_pageGeometry;
    QSizeF m_requestedSize;
    QSizeF m_pageImageSize;
    Qt::Orientation m_orientation;
    QSharedPointer<BasePage> m_pageSource;
    qreal m_scale;
    qreal m_imageScale;
    DocumentMapper *m_mapper;
    QHash<int, MultiPageTile *> m_tilesMap;
    BackgroundPage *m_backgroundPage;
    qreal m_maxTileZ;
    QRectF m_visibleArea;
    quint16 m_tileSize;
    bool m_annotationsPaint;
    bool m_notesPaint;
    bool m_cellSelectedPaint;
    bool m_allTilesReady;
    QList<BaseDocItem *> m_annotationsItems;
    QList<BaseDocItem *> m_commentsItems;
    QList<BaseDocItem *> m_highlightedItems;
    QList<BaseDocItem *> m_currentCellItems;
    QColor m_highLightColor;
    QQuickItem *m_cursor;
};
