/*******************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
** This file is part of the OfficeViewer project.
**
** Redistribution and use in source and binary forms,
** with or without modification, are permitted provided
** that the following conditions are met:
**
** * Redistributions of source code must retain the above copyright notice,
**   this list of conditions and the following disclaimer.
** * Redistributions in binary form must reproduce the above copyright notice,
**   this list of conditions and the following disclaimer
**   in the documentation and/or other materials provided with the distribution.
** * Neither the name of the copyright holder nor the names of its contributors
**   may be used to endorse or promote products derived from this software
**   without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
** THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
** FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
** EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*******************************************************************************/


#include <QMutexLocker>
#include <QThread>
#include <QTimer>

#include "debug.h"
#include "basepage.h"

#include "bitmaploadworker.h"

QMutex BitmapLoaderWorker::m_mutex;

BitmapLoaderWorker::BitmapLoaderWorker(QSharedPointer<BasePage> p, qreal scaleX, qreal scaleY, int flags, qreal zoom , QPointF bias, qreal pageScale):
    m_page(p),
    m_scaleX(scaleX),
    m_scaleY(scaleY),
    m_flags(flags),
    m_zoom(zoom),
    m_bias(bias),
    m_pageScale(pageScale),
    m_isCanceled(false),
    m_loaded(false)
{
}

BitmapLoaderWorker::~BitmapLoaderWorker()
{
    cancel();
    qCDebug(lcDocviewerRender) << "Destroy loader page index:" << m_page->pageIndex();
#ifdef DEBUG_MODE
    m_page->stopBitmapPart(m_scaleX, m_scaleY, m_zoom, m_bias);
#endif
}

void BitmapLoaderWorker::run() {
    if (!m_page) {
        emit done({  });
        return;
    }

    if (m_isCanceled) {
        emit done({  });
        return;
    }

    setAutoDelete(false);
    connect(&m_watcher, &QFutureWatcher<QImage>::finished, this, &BitmapLoaderWorker::_getResult);
    QMutexLocker lock(&m_mutex);
    m_watcher.setFuture(m_page->bitmapPart(m_scaleX, m_scaleY, m_flags, m_zoom, m_bias, m_pageScale));

    // If after 30 seconds no data has been received, then delete the loader
    QTimer::singleShot(30 * 1000, this, &BitmapLoaderWorker::_getResult);
}

void BitmapLoaderWorker::cancel()
{
    if (m_watcher.isRunning())
        m_watcher.cancel();

    if (m_isCanceled)
        return;

    if (m_page && !m_loaded)
        m_page->stopBitmapPart(m_scaleX, m_scaleY, m_zoom, m_bias);
    else
        qCDebug(lcDocviewerRender) << "Not cancel";

    m_isCanceled = true;
    setAutoDelete(true);
    deleteLater();
}

void BitmapLoaderWorker::_getResult()
{

    if (m_watcher.isFinished() && !m_watcher.isCanceled())
        emit done(m_watcher.result());
    else
        emit done({  });

    m_loaded = true;
    setAutoDelete(true);
    deleteLater();
}
