/*******************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
** This file is part of the OfficeViewer project.
**
** Redistribution and use in source and binary forms,
** with or without modification, are permitted provided
** that the following conditions are met:
**
** * Redistributions of source code must retain the above copyright notice,
**   this list of conditions and the following disclaimer.
** * Redistributions in binary form must reproduce the above copyright notice,
**   this list of conditions and the following disclaimer
**   in the documentation and/or other materials provided with the distribution.
** * Neither the name of the copyright holder nor the names of its contributors
**   may be used to endorse or promote products derived from this software
**   without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
** THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
** FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
** EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*******************************************************************************/


import QtQuick 2.3
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import "components"

Item {
    id: root

    property int minimumValue
    property int maximumValue
    property int currentValue
    property var pageNames
    property bool tabsNavigation
    property bool selectionMode
    property string url

    signal currentPageChanged(var newPageNumber)
    signal copySelectionText
    signal shareSelectionText
    signal clearSelectionText

    height: content.height

    MouseArea {
        // Trapping mouse events on an element
        anchors.fill: parent
    }

    Rectangle {
        id: navigationToolbar

        color: Qt.tint(palette.colorScheme === Theme.LightOnDark ? "black" : "white",
                              Theme.rgba(palette.primaryColor, Theme.opacityFaint))
        anchors.fill: parent

        Loader {
            id: content

            width: parent.width
            sourceComponent: url ? urlToolbar : selectionMode ? slectionToolbar : tabsNavigation ? tabsToolbar : sliderToolbar
        }
    }
    Component {
        id: sliderToolbar

        Item {
            height: slider.height + Theme.paddingLarge * 2
            Slider {
                id: slider

                property bool blockSignals

                onDownChanged: {
                    if (blockSignals)
                        return

                    if (!down) {
                        currentPageShowTimer.stop()
                        currentPageIndex.opacity = 0.0
                        currentPageChanged(value)
                    } else {
                        currentPageShowTimer.start()
                    }
                }

                stepSize: 1
                valueText: ""
                minimumValue: root.minimumValue
                maximumValue: root.maximumValue
                value: root.currentValue

                secondaryHighlightColor: color
                animateValue: false
                leftMargin: Theme.horizontalPageMargin
                rightMargin: Theme.horizontalPageMargin

                anchors {
                    left: parent.left
                    leftMargin: Theme.horizontalPageMargin
                    right: parent.right
                    rightMargin: Theme.horizontalPageMargin
                    verticalCenter: parent.verticalCenter
                }

                Text {
                    id: currentPageIndex

                    x: slider._highlightX + slider._highlightPadding + (slider.colorScheme === Theme.LightOnDark ? 1.5 : 1) * 2 * slider._glassItemPadding - width * 0.5
                    y: slider._progressBarItem.y - height
                    opacity: 0.0
                    color: palette.primaryColor
                    text: slider.value

                    font.pixelSize: Theme.fontSizeMedium

                    Behavior on opacity {
                        NumberAnimation {
                            duration: 200
                        }
                    }
                }

                Text {
                    color: palette.secondaryColor
                    text: "/"
                    anchors { top: slider.bottom; topMargin: -height * 0.75; horizontalCenter: parent.horizontalCenter }
                    font.pixelSize: Theme.fontSizeMedium

                    Text {
                        color: palette.primaryColor
                        text: slider.value
                        anchors.right: parent.left
                        font.pixelSize: parent.font.pixelSize
                    }

                    Text {
                        color: parent.color
                        text: slider.maximumValue
                        font.pixelSize: parent.font.pixelSize
                        anchors.left: parent.right
                    }
                }

                Timer {
                    id: currentPageShowTimer

                    onTriggered: currentPageIndex.opacity = 1.0

                    interval: 200
                }
            }
            Connections {
                target: root
                onCurrentValueChanged: {
                    slider.blockSignals = true
                    slider.value = root.currentValue
                    slider.blockSignals = false
                }
            }
        }
    }

    Component {
        id: tabsToolbar

        TabView {
            id: tabView
            property var pageNames: root.pageNames
            property bool blockSignals

            onPageNamesChanged: {
                namesModel.clear()
                for (var i in pageNames) namesModel.append({"title": pageNames[i]})
            }

            onCurrentIndexChanged: if (!blockSignals) currentPageChanged(currentIndex + 1)

            model: root.maximumValue
            delegate: Item {}

            footer: TabBar {
                id: tabBar

                model: namesModel
                delegate: TabButton {
                    title: model.title
                    Rectangle {
                        anchors.fill: parent
                        color: navigationToolbar.color
                        z: -1
                    }
                }
                Component.onCompleted: tabView.height = Qt.binding(function() { return height } )
            }

            ListModel {
                id: namesModel
            }

            Connections {
                target: root
                onCurrentValueChanged: {
                    tabView.blockSignals = true
                    if (tabsNavigation) tabView.moveTo(root.currentValue - 1)
                    tabView.blockSignals = false
                }
            }
        }
    }
    Component {
        id: urlToolbar

        Item {
            height: linkedLabel.height + Theme.paddingLarge * 2
            LinkedLabel {
                id: linkedLabel

                anchors.centerIn: parent
                x: Theme.horizontalPageMargin
                width: parent.width - 2*x
                visible: root.url
                plainText: root.url
                color: highlighted ? Theme.highlightColor : Theme.primaryColor
                linkColor: Theme.highlightColor
                shortenUrl: true
                horizontalAlignment: lineCount > 1 ? Text.AlignLeft : Text.AlignHCenter
            }
        }
    }
    Component {
        id: slectionToolbar

        TextSelectionToolbar {
            onShare: shareSelectionText()
            onCopy: copySelectionText()
            onClear: clearSelectionText()
        }
    }
}
