/*******************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
** This file is part of the OfficeViewer project.
**
** Redistribution and use in source and binary forms,
** with or without modification, are permitted provided
** that the following conditions are met:
**
** * Redistributions of source code must retain the above copyright notice,
**   this list of conditions and the following disclaimer.
** * Redistributions in binary form must reproduce the above copyright notice,
**   this list of conditions and the following disclaimer
**   in the documentation and/or other materials provided with the distribution.
** * Neither the name of the copyright holder nor the names of its contributors
**   may be used to endorse or promote products derived from this software
**   without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
** THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
** FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
** EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*******************************************************************************/


import QtQuick 2.0
import Sailfish.Silica 1.0

Item {
    id: root

    property var document
    property color color: Theme.primaryColor
    property alias backgroundColor: background.color
    property alias backgroundVisible: background.visible
    property bool opened: true
    property bool searchMode
    property bool allowedVisibleToolbar: true

    signal openMenu
    signal searchBackward(string searchString)
    signal searchForward(string searchString)
    signal stopSearch

    onSearchModeChanged: if (!searchMode) opened = true

    width: parent.width
    height: content.height
    y: searchMode || opened ? 0 : -height

    Behavior on y { NumberAnimation { duration: 200 } }


    Loader {
        id: content

        width: parent.width
        sourceComponent: searchMode ? itemSearch : itemHeader
    }

    Rectangle {
        id: background

        anchors.fill: parent
        z: -1
        color: Qt.tint(palette.colorScheme === Theme.LightOnDark ? "black" : "white",
                       Theme.rgba(palette.primaryColor, Theme.opacityFaint))
    }

    Component {
        id: itemHeader

        Item {
            height: pageHeader.height

            PageHeader {
                id: pageHeader

                width: parent.width
                title: document.fileName
                titleColor: Theme.highlightColor
                rightMargin: iconMenu.width
                leftMargin: Theme.horizontalPageMargin + Theme.paddingSmall
            }

            IconButton {
                id: iconMenu

                onClicked: openMenu()

                anchors.verticalCenter: parent.verticalCenter
                anchors.right: parent.right
                icon.source: "image://theme/icon-m-menu"
                opacity: root.enabled ? 1.0 : Theme.opacityHigh

                Behavior on opacity { FadeAnimator {}}
            }            
            Component.onCompleted: root.allowedVisibleToolbar = true
        }
    }

    Component {
        id: itemSearch

        Item {
            height: searchField.height

            SearchField {
                id: searchField

                property bool allowedVisibleToolbar: (orientation & Orientation.PortraitMask) || !activeFocus

                width: parent.width
                canHide: false
                placeholderText: qsTrId("docviewer-la-search_by_doc")
                inputMethodHints: Qt.ImhNoAutoUppercase | Qt.ImhPreferLowercase | Qt.ImhNoPredictiveText
                EnterKey.iconSource: "image://theme/icon-m-enter-accept"
                EnterKey.onClicked: {
                    searchField.focus = false
                    searchForward(searchField.text)
                }
                leftItem: IconButton {
                    onClicked: {
                        searchMode = false
                        stopSearch()
                    }
                    icon.source: "image://theme/icon-m-cancel"
                }
                rightItem: Item {
                    width: (iconRemove.enabled ? iconRemove.width : 0) + (iconUp.enabled ? iconUp.width : 0) + (iconDown.enabled ? iconDown.width : 0)
                    height: iconRemove.height

                    IconButton {
                        id: iconUp

                        onClicked: searchBackward(searchField.text)

                        anchors.left: parent.left
                        icon.source: "image://theme/icon-m-up"
                        enabled: !searchField.focus && searchField.text.length > 0
                        opacity: enabled ? 1.0 : 0.0
                        Behavior on opacity { FadeAnimation {} }
                    }

                    IconButton {
                        id: iconDown

                        onClicked: searchForward(searchField.text)

                        anchors.left: iconUp.right
                        icon.source: "image://theme/icon-m-down"
                        enabled: !searchField.focus && searchField.text.length > 0
                        opacity: enabled ? 1.0 : 0.0
                        Behavior on opacity { FadeAnimation {} }

                    }

                    IconButton {
                        id: iconRemove

                        onClicked: {
                            if (searchField.canHide && searchField.text.length === 0) {
                                searchField.hideClicked()
                            } else {
                                searchField.text = ""
                            }
                        }

                        anchors.left: parent.left
                        icon.source: searchField.canHide && searchField.text.length === 0
                                     ? "image://theme/icon-m-input-remove"
                                     : "image://theme/icon-m-clear"

                        enabled: searchField.focus && searchField.text.length > 0

                        opacity: enabled ? 1.0 : 0.0
                        Behavior on opacity { FadeAnimation {} }
                    }
                }

                onHideClicked: searchMode = false
                onAllowedVisibleToolbarChanged: root.allowedVisibleToolbar = allowedVisibleToolbar
                Component.onCompleted: {
                    forceActiveFocus()
                }
            }
        }
    }
}
