/*******************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
** This file is part of the OfficeViewer project.
**
** Redistribution and use in source and binary forms,
** with or without modification, are permitted provided
** that the following conditions are met:
**
** * Redistributions of source code must retain the above copyright notice,
**   this list of conditions and the following disclaimer.
** * Redistributions in binary form must reproduce the above copyright notice,
**   this list of conditions and the following disclaimer
**   in the documentation and/or other materials provided with the distribution.
** * Neither the name of the copyright holder nor the names of its contributors
**   may be used to endorse or promote products derived from this software
**   without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
** THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
** FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
** EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*******************************************************************************/


#include "fileinfo.h"

#include <QtCore/QDir>
#include <QtCore/QFileInfo>
#include <QtCore/QMimeDatabase>
#include <QtCore/QMimeType>

class FileInfo::Private
{
public:
    Private() {}

    void resolvePath();

    QString source;

    QUrl path;

    QFileInfo fileInfo;
    QMimeType mimeType;
};

FileInfo::FileInfo(QObject *parent)
    : QObject(parent)
    , d(new Private)
{}

FileInfo::~FileInfo()
{
    delete d;
}

QString FileInfo::source() const
{
    return d->source;
}

void FileInfo::setSource(const QString &source)
{
    if (source != d->source) {
        d->source = source;

        d->resolvePath();

        emit sourceChanged();
    }
}

QString FileInfo::fileName() const
{
    return d->fileInfo.fileName();
}

QUrl FileInfo::fileUrl() const
{
    return d->path;
}

QString FileInfo::filePath() const
{
    return d->fileInfo.filePath();
}

qint64 FileInfo::fileSize() const
{
    return d->fileInfo.size();
}

QString FileInfo::mimeType() const
{
    return d->mimeType.name();
}

QString FileInfo::mimeTypeComment() const
{
    return d->mimeType.comment();
}

QDateTime FileInfo::modifiedDate() const
{
    return d->fileInfo.lastModified();
}

QString FileInfo::path() const
{
    return d->fileInfo.path();
}

void FileInfo::Private::resolvePath()
{
    path = QUrl(source);
    if (path.isEmpty())
        path = QUrl::fromLocalFile(source);

    if (path.isRelative())
        path = QUrl::fromLocalFile(QDir::current().absoluteFilePath(source));

    fileInfo = QFileInfo(path.toLocalFile());
    QMimeDatabase db;
    mimeType = db.mimeTypeForFile(fileInfo);
}

QString FileInfo::baseName() const
{
    return d->fileInfo.baseName();
}
