/*******************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
** This file is part of the OfficeViewer project.
**
** Redistribution and use in source and binary forms,
** with or without modification, are permitted provided
** that the following conditions are met:
**
** * Redistributions of source code must retain the above copyright notice,
**   this list of conditions and the following disclaimer.
** * Redistributions in binary form must reproduce the above copyright notice,
**   this list of conditions and the following disclaimer
**   in the documentation and/or other materials provided with the distribution.
** * Neither the name of the copyright holder nor the names of its contributors
**   may be used to endorse or promote products derived from this software
**   without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
** THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
** FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
** EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*******************************************************************************/


import QtQuick 2.2
import Sailfish.Silica 1.0
import Sailfish.Share 1.0
import Sailfish.Pickers 1.0
import ru.omp.documentprovider.libreoffice 1.0
import ru.omp.docviewer 1.0

Page {
    id: root

    property string path
    property QtObject trackerDocumentProvider
    property alias documentBaseName: fileInfo.fileName
    property bool visibleToolbar
    property alias _docView: docView

    allowedOrientations: Orientation.All
    objectName: "documentPage"
    backNavigation: docHeader.opened && !docHeader.searchMode

    onStatusChanged: {
        if (pageStack.currentPage.objectName === objectName) {
            if (status === PageStatus.Active) {
                docView.path = path
            }
        }
    }

    ProgressLoad {
        visible: docView.status === LoDocument.Loading
        document: loDocument
    }

    DocumentHeader {
        id: docHeader

        property real startedContentY

        z: docView.z + 1
        width: root.width
        document: loDocument
        backgroundVisible: loDocument.status === LoDocument.Ready
        enabled: loDocument.status === LoDocument.Ready

        onOpenMenu: popupMenu.active = true
        onSearchBackward: docView.startSearch(searchString, LoDocument.SearchBackward)
        onSearchForward: docView.startSearch(searchString, LoDocument.SearchForward)
        onStopSearch: docView.stopSearch()
    }

    DocumentView {
        id: docView

        property bool preparePrinting
        property bool exporting

        function updateVisibleHeader() {
            if (contentHeight === height)
                docHeader.opened = true
        }

        documentProvider: loDocument
        topMargin: docHeader.height
        clip: true
        width: parent.width
        catchBound: docHeader.height
        annotationsPaint: true
        notesPaint: true
        orientation: root.orientation
        anchors.horizontalCenter: parent.horizontalCenter

        onClicked: {
            if (status === LoDocument.Ready && loDocument.documentType !== LoDocumentProvider.Spreadsheet) {
                visibleToolbar = !visibleToolbar
                toolbar.url = ""
            }
        }
        onClickedUrl: {
            popupMenuHyperlink.url = url
            popupMenuHyperlink.active = true
        }
        onClickedCell: {
            popupMenuHyperlink.cellContent = cellContent
            popupMenuHyperlink.active = true
        }

        onNoteActivate: pageStack.push(Qt.resolvedUrl("CommentPage.qml"), { text: text, author: author, dateTime: dateTime })

        onVerticalVelocityChanged: {
            if (docHeader.enabled && verticalVelocity < 0)
                docHeader.opened = true
            else if (docHeader.enabled && verticalVelocity > 0)
                docHeader.opened = false

            updateVisibleHeader()
        }

        onMovingChanged: {
            if (moving) {
                docHeader.startedContentY = contentY
            } else {
                if (docHeader.startedContentY - contentY > docHeader.height)
                    docHeader.opened = true

                if (contentY - docHeader.startedContentY > docHeader.height)
                    docHeader.opened = false
            }

            updateVisibleHeader()
        }

        onContentHeightChanged: {
            updateVisibleHeader()
        }

        Component.onCompleted: {
            height = Qt.binding(function() { return root.height })
        }
    }

    ProcessingIndicator {
        y: docHeader.y + docHeader.height
        visible: loDocument.status === LoDocument.Ready
        processing: loDocument.processing || docView.preparePrinting || docView.exporting
        text: docView.exporting
              //% "Exporting"
              ? qsTrId("docviewer-la-exporting")
              : docView.preparePrinting
                  //% "Preparing for printing"
                  ? qsTrId("docviewer-la-preparing_print")
                  //% "Loading"
                  : qsTrId("docviewer-la-processing")
        z: 1
    }

    ZoomIndicator {
        anchors {
            horizontalCenter: parent.horizontalCenter
            bottom: parent.bottom
            bottomMargin: toolbar.visible ? toolbar.height + Theme.paddingMedium : Theme.paddingMedium
        }

        zoom: docView.zoom
    }

    OfficeToolbar {
        id: toolbar

        onCurrentPageChanged: docView.goToPage(newPageNumber - 1)
        onClearSelectionText: docView.resetSelection()
        onCopySelectionText: {
            Clipboard.text = docView.selectionText
            //% "Сopied"
            Notices.show(qsTrId("docviewer-me-copied"), Notice.Long, Notice.Center)
            console.log("Copied:", docView.selectionText)
        }
        onShareSelectionText: shareAction.shareText(docView.selectionText)

        visible: loDocument.status === LoDocument.Ready
                 && ((visibleToolbar && docHeader.allowedVisibleToolbar)
                     || loDocument.documentType === LoDocumentProvider.Spreadsheet
                     || docView.selectionMode)
        width: parent.width
        tabsNavigation: loDocument.documentType === LoDocumentProvider.Spreadsheet
        minimumValue: loDocument.status === LoDocument.Ready ? 1 : 0
        maximumValue: loDocument.status === LoDocument.Ready ? docView.count : 2
        currentValue: docView.currentIndex + 1
        pageNames: loDocument.pageNames
        selectionMode: docView.selectionMode
        anchors.bottom: parent.bottom
    }

    PopUpMenu {
        id: popupMenu

        anchors.fill: parent
        active: false
        z: docView.z + 1
        verticalPosition: Qt.AlignTop
        horizontalPosition: Qt.AlignRight

        menuItem: Component {
            PopUpMenuItem {
                onShareClicked: shareAction.shareFile()
                onInfoClicked: {
                    pageStack.push(Qt.resolvedUrl("DetailsPage.qml"), {
                                                        source: loDocument.path,
                                                        indexCount: loDocument.count,
                                                        mimeType: fileInfo.mimeType
                                                  },
                                                  PageStackAction.Animated)
                }
                onSearchClicked: docHeader.searchMode = true
                onExportToPdf: pageStack.animatorPush(folderPickerExport, { "exportPdf": true })
                onPrintClicked: {
                    docView.saveDocumentAs("%1/%2.pdf".arg("/tmp").arg(fileInfo.baseName), "pdf")
                    docView.preparePrinting = true
                }

                onSaveAsClicked: pageStack.animatorPush(folderPickerExport, { "exportPdf": false })

                enabledSearch: loDocument.documentType === LoDocumentProvider.Text
                enabledExportPdf: !docView.exporting && !docView.preparePrinting
            }
        }
        onClosed: active = false
    }

    PopUpMenu {
        id: popupMenuHyperlink

        property string url
        property string cellContent

        readonly property int widthRatioForUrl: 18
        readonly property int widthRatioForCellContent: 22
        readonly property int widthRatioForCellContentLarge: 34

        onUrlChanged: if (cellContent) cellContent = ""
        onCellContentChanged: if (url) url = ""

        anchors.fill: parent
        verticalPosition: url ? Qt.AlignVCenter : Qt.AlignBottom

        widthRatio: url ? widthRatioForUrl : Screen.sizeCategory > Screen.Medium ? widthRatioForCellContentLarge : widthRatioForCellContent
        minHeight: url ? Theme.paddingLarge * heightRatio : Theme.fontSizeMedium + Theme.paddingLarge * 2 + Theme.paddingSmall
        active: false
        headerItemVisible: url ? false : true
        z: docView.z + 1        
        menuItem: url ? componentHyperlink : componentCellContent
        footer: url ? undefined : componentCellFooter

        Component {
            id: componentCellFooter

            PopUpMenuCellContentFooter {
                text: popupMenuHyperlink.cellContent
                onCloseMenu: popupMenuHyperlink.closed()
            }
        }

        Component {
            id: componentHyperlink

            PopUpMenuHyperlink {
                url: popupMenuHyperlink.url
                onCloseMenu: popupMenuHyperlink.closed()
            }
        }
        Component {
            id: componentCellContent

            PopUpMenuCellContent {
                isViewFull: !popupMenuHyperlink.isShowFull
                content: popupMenuHyperlink.cellContent
                onCloseMenu: popupMenuHyperlink.closed()
            }
        }
        onClosed: active = false
    }

    Loader {
        id: errorLoader

        onLoaded: item.provider = loDocument
        onSourceChanged: {
            if (loDocument.status === LoDocument.Error) {
                popTimer.start()
            }
        }

        anchors.fill: parent
        source: loDocument.status === LoDocument.Error ? Qt.resolvedUrl("FullscreenError.qml") : ""

        Timer {
            id: popTimer

            interval: 5000
            onTriggered: pageStack.pop()
        }
    }

    FileInfo {
        id: fileInfo

        source: root.path
    }

    LoDocument {
        id: loDocument

        onShowInfo: Notices.show(info, Notice.Long, Notice.Center)
        onSaveAsStatusChanged: {
            if (docView.preparePrinting) {
                if (status) {
                    pageStack.animatorPush("Sailfish.Pickers.PrinterPickerPage", { "printPath":  "%1/%2.pdf".arg("/tmp").arg(fileInfo.baseName) })
                } else {
                    //% "Failed to prepare for printing"
                    Notices.show(qsTrId("docviewer-la-prepare_for_printing"), Notice.Long, Notice.Center)
                }

                docView.preparePrinting = false
            } else {
                if (status) {
                    //: Export to document in pdf format
                    //% "Exported successfully"
                    Notices.show(qsTrId("docviewer-la-exported_successfully"), Notice.Long, Notice.Center)
                } else {
                    //: Failed to export to document in pdf format
                    //% "Failed to export"
                    Notices.show(qsTrId("docviewer-la-failed_export"), Notice.Long, Notice.Center)
                }
            }

            docView.exporting = false
        }
    }

    ShareAction {
        id: shareAction


        function shareFile() {

            resources = [fileInfo.fileUrl]
            mimeType = fileInfo.mimeType
        }

        function shareText(text) {
            resources = [{
                             "name": text,
                             "data": text,
                             "type": "text/plain",
                             "status": text,
                         }]
            mimeType = "text/plain"
            trigger()
        }
        //: Header for text sharing
        //% "Share"
        title: qsTrId("docviewer-he-share")
    }

    Component {
        id: folderPickerExport

        FolderPickerPage {
            property bool exportPdf: true

            showSystemFiles: false

            dialogTitle: exportPdf
                         //% "Export to PDF"
                         ? qsTrId("docviewer-ti-export_to_pdf")
                           //% "Save as"
                         : qsTrId("docviewer-ti-save_as")

            onSelectedPathChanged: {
                if (exportPdf) {
                    exportLater.exportFile("%1/%2.pdf".arg(selectedPath).arg(fileInfo.baseName), ".pdf")
                } else {
                    pageStack.animatorPush(saveAsDialog, {
                                               "path" : selectedPath,
                                               "baseName" : fileInfo.baseName,
                                               "sourceMimeType" : fileInfo.mimeType
                                           })
                }
            }
        }
    }

    Component {
        id: saveAsDialog

        SaveAsDialog {
            acceptDestination: root
            acceptDestinationAction: PageStackAction.Pop

            onAccepted: {
                console.info("Save As ", "%1/%2%3".arg(path).arg(baseName).arg(fileExtension))
                exportLater.exportFile("%1/%2%3".arg(path).arg(baseName).arg(fileExtension), fileExtension)
            }
        }
    }

    Timer {
        id: exportLater

        property string path
        property string format

        function exportFile(filePath, fileExtension) {
            path = filePath
            format = fileExtension.replace('.', '')
            start()
            docView.exporting = true
        }

        interval: 500
        onTriggered: docView.saveDocumentAs(path, format)
    }
}
